<?php
namespace Newland\Toubiz\Sync\Neos\Domain\Model;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Neos\Flow\Annotations as Flow;

/**
 * An event.
 *
 * @Flow\Entity
 * @todo adapt http://flowframework.readthedocs.io/en/stable/TheDefinitiveGuide/PartIII/Persistence.html
 */
class Event extends AbstractEntity
{
    /**
     * @var string Title of the event.
     */
    protected $title;

    /**
     * @var string Description of the event.
     * @ORM\Column(type="text", nullable=true)
     */
    protected $description;

    /**
     * @var string Information about admission.
     * @ORM\Column(type="text", nullable=true)
     */
    protected $admission;

    /**
     * @var bool
     */
    protected $isHighlight = false;

    /**
     * @var bool
     */
    protected $isTipp = false;

    /**
     * @var \DateTime Date and time of first day's beginning.
     */
    protected $beginsAt;

    /**
     * @var \DateTime Date and time of last day's ending.
     */
    protected $endsAt;

    /**
     * @var string URI to buy a ticket.
     * @ORM\Column(type="text", nullable=true)
     */
    protected $ticketUri;

    /**
     * @ORM\ManyToMany(targetEntity="Newland\Toubiz\Sync\Neos\Domain\Model\Category", inversedBy="events", fetch="EAGER")
     * @var Collection<Category>
     */
    protected $categories;

    /**
     * @ORM\OneToMany(mappedBy="event")
     * @ORM\OrderBy({"beginsAt"="ASC"})
     * @var Collection<EventDate>
     */
    protected $eventDates;

    /**
     * @ORM\ManyToMany(targetEntity="Newland\Toubiz\Sync\Neos\Domain\Model\Medium", inversedBy="events", fetch="LAZY")
     * @var Collection<Media>
     */
    protected $media;

    /**
     * @ORM\ManyToMany(targetEntity="Newland\Toubiz\Sync\Neos\Domain\Model\Feature", inversedBy="events")
     * @var Collection<Feature>
     */
    protected $features;

    /**
     * @var \DateTime
     */
    protected $updatedAt;

    /**
     * @var Address
     * @ORM\ManyToOne(targetEntity="Newland\Toubiz\Sync\Neos\Domain\Model\Address")
     */
    protected $location;

    /**
     * @var Address
     * @ORM\ManyToOne(targetEntity="Newland\Toubiz\Sync\Neos\Domain\Model\Address")
     */
    protected $organizer;

    /**
     * Class constructor.
     *
     * This constructor is called before hydration and dependency
     * injection is happening.
     *
     * @return void
     */
    public function __construct()
    {
        $this->categories = new ArrayCollection;
        $this->eventDates = new ArrayCollection;
        $this->features = new ArrayCollection;
        $this->media = new ArrayCollection;
        $this->updatedAt = new \DateTime;
    }

    /**
     * @param string $title
     * @return void
     */
    public function setTitle($title)
    {
        $this->title = $title;
    }

    /**
     * @return string
     */
    public function getTitle()
    {
        return $this->title;
    }

    /**
     * @param string $description
     * @return void
     */
    public function setDescription($description)
    {
        $this->description = $description;
    }

    /**
     * @return string
     */
    public function getDescription()
    {
        return $this->description;
    }

    /**
     * @param string $admission
     * @return void
     */
    public function setAdmission($admission)
    {
        $this->admission = $admission;
    }

    /**
     * @return string
     */
    public function getAdmission()
    {
        return $this->admission;
    }

    /**
     * @param \DateTime $updatedAt
     * @return void
     */
    public function setUpdatedAt(\DateTime $updatedAt)
    {
        $this->updatedAt = $updatedAt;
    }

    /**
     * @return \DateTime
     */
    public function getUpdatedAt()
    {
        return $this->updatedAt;
    }

    /**
     * @param bool $highlight
     * @return void
     */
    public function setIsHighlight($highlight)
    {
        $this->isHighlight = (bool)$highlight;
    }

    /**
     * @return bool
     */
    public function getIsHighlight()
    {
        return $this->isHighlight;
    }

    /**
     * @param bool $tipp
     * @return void
     */
    public function setIsTipp($tipp)
    {
        $this->isTipp = (bool)$tipp;
    }

    /**
     * @return bool
     */
    public function getIsTipp()
    {
        return $this->isTipp;
    }

    /**
     * @param \DateTime $beginsAt
     * @return void
     */
    public function setBeginsAt(\DateTime $beginsAt)
    {
        $this->beginsAt = $beginsAt;
    }

    /**
     * @return \DateTime
     */
    public function getBeginsAt()
    {
        return $this->beginsAt;
    }

    /**
     * @param string $ticketUri
     * @return void
     */
    public function setTicketUri($ticketUri)
    {
        $this->ticketUri = $ticketUri;
    }

    /**
     * @return string
     */
    public function getTicketUri()
    {
        return $this->ticketUri;
    }

    /**
     * @return bool
     */
    public function getBeginsAtSpecificTime()
    {
        return ((int)$this->beginsAt->format('H') > 0);
    }

    /**
     * @param \DateTime $endsAt
     * @return void
     */
    public function setEndsAt(\DateTime $endsAt)
    {
        $this->endsAt = $endsAt;
    }

    /**
     * @return \DateTime
     */
    public function getEndsAt()
    {
        return $this->endsAt;
    }

    /**
     * @param Address $location
     * @return void
     */
    public function setLocation(Address $location = null)
    {
        $this->location = $location;
    }

    /**
     * @return Address
     */
    public function getLocation()
    {
        return $this->location;
    }

    /**
     * @param Address $organizer
     * @return void
     */
    public function setOrganizer(Address $organizer = null)
    {
        $this->organizer = $organizer;
    }

    /**
     * @return Address
     */
    public function getOrganizer()
    {
        return $this->organizer;
    }

    /**
     * @param Collection $categories
     * @return void
     */
    public function setCategories(Collection $categories)
    {
        $this->categories = $categories;
    }

    /**
     * @return Collection
     */
    public function getCategories()
    {
        return $this->categories;
    }

    /**
     * @param Collection $eventDates
     * @return void
     */
    public function setEventDates(Collection $eventDates)
    {
        $this->eventDates = $eventDates;
    }

    /**
     * @return Collection
     */
    public function getEventDates()
    {
        return $this->eventDates;
    }

    /**
     * Get event dates having a note.
     *
     * @return array
     */
    public function getAnnotatedEventDates()
    {
        return array_filter(
            $this->eventDates->toArray(),
            function ($event) {
                return !empty($event->getNote());
            }
        );
    }

    /**
     * Get event dates in groups.
     *
     * Each event date record represents one single day
     * which may be grouped into a date group in case
     * a bunch of days represents multiple days (like a "week").
     *
     * @return array
     */
    public function getUpcomingEventDateGroups()
    {
        $groups = [];
        $count = count($this->eventDates);
        foreach ($this->eventDates->toArray() as $currentDate)
        {
            if ($currentDate->getIsInThePast()) {
                continue;
            }

            if (!isset($previousDate)) {
                // Initialize on a new group search and skip current iteration.
                $groupStartDate = $currentDate;
                $previousDate = $currentDate;
                continue;
            }

            $interval = $currentDate->getBeginsAt()->diff($previousDate->getBeginsAt());
            if ($interval->format('%d') > 1) {
                // On a difference bigger than one day, a group ends.
                $groupDate = new EventDate;
                $groupDate->setBeginsAt($groupStartDate->getBeginsAt());
                $groupDate->setEndsAt($previousDate->getEndsAt());
                $groups[] = $groupDate;

                // Reset previous date to ensure new initialization on next iteration.
                unset($groupStartDate, $previousDate, $groupDate);
                continue;
            }

            // Assign previou date for next iteration.
            $previousDate = $currentDate;
        }

        /*
         * There is either an unclosed group date or the last iteration does
         * not count its current date in, hence it must be added manually.
         */
        if (!isset($groupDate)) {
            $groupDate = new EventDate;
            $groupDate->setBeginsAt($currentDate->getBeginsAt());
        }
        $groupDate->setEndsAt($currentDate->getEndsAt());
        $groups[] = $groupDate;

        return $groups;
    }

    /**
     * @return bool
     */
    public function getEndsOnSameDay()
    {
        return ($this->beginsAt->format('Y-m-d') == $this->endsAt->format('Y-m-d'));
    }

    /**
     * @param Collection $media
     * @return void
     */
    public function setMedia(Collection $media)
    {
        $this->media = $media;
    }

    /**
     * @return Collection
     */
    public function getMedia()
    {
        return $this->media;
    }

    /**
     * @param Collection $features
     * @return void
     */
    public function setFeatures(Collection $features)
    {
        $this->features = $features;
    }

    /**
     * @return Collection
     */
    public function getFeatures()
    {
        return $this->features;
    }
}
