<?php
namespace Newland\Toubiz\Sync\Neos\Domain\Model;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;

/**
 * An article.
 *
 * This represents several article types, like a "point of interest",
 * a lodging, a tour, etc.
 *
 * @Flow\Entity
 */
class Article extends AbstractEntity implements \JsonSerializable
{
    /**
     * @var int Type of the article.
     */
    protected $mainType;

    /**
     * @var string Name of the article.
     */
    protected $name;

    /**
     * @ORM\Column(type="text", nullable=true)
     * @var string Abstract for this article.
     */
    protected $abstract;

    /**
     * @ORM\Column(type="text", nullable=true)
     * @var string Descriptive text for this article.
     */
    protected $description;

    /**
     * @ORM\Column(nullable=true)
     * @var string
     */
    protected $street;

    /**
     * @ORM\Column(nullable=true)
     * @var string
     */
    protected $zip;

    /**
     * @ORM\Column(nullable=true)
     * @var string
     */
    protected $city;

    /**
     * @ORM\Column(nullable=true)
     * @var string
     */
    protected $country;

    /**
     * @ORM\Column(nullable=true)
     * @var float
     */
    protected $latitude;

    /**
     * @ORM\Column(nullable=true)
     * @var float
     */
    protected $longitude;

    /**
     * @ORM\ManyToMany(targetEntity="Newland\Toubiz\Sync\Neos\Domain\Model\Address", inversedBy="articles", fetch="EAGER")
     * @var Collection<Address>
     */
    protected $addresses;

    /**
     * @ORM\ManyToMany(targetEntity="Newland\Toubiz\Sync\Neos\Domain\Model\Category", inversedBy="articles", fetch="EAGER")
     * @var Collection<Category>
     */
    protected $categories;

    /**
     * @ORM\ManyToMany(targetEntity="Newland\Toubiz\Sync\Neos\Domain\Model\Medium", inversedBy="articles", fetch="LAZY")
     * @var Collection<Media>
     */
    protected $media;

    /**
     * @ORM\ManyToMany(targetEntity="Newland\Toubiz\Sync\Neos\Domain\Model\File", inversedBy="articles", fetch="LAZY")
     * @var Collection<File>
     */
    protected $files;

    /**
     * @ORM\Column(type="array", nullable=true)
     * @var array
     */
    protected $attributes;

    /**
     * @ORM\Column(nullable=true)
     * @var string
     */
    protected $facebookUri;

    /**
     * @ORM\Column(nullable=true)
     * @var string
     */
    protected $twitterUri;

    /**
     * @ORM\Column(nullable=true)
     * @var string
     */
    protected $instagramUri;

    /**
     * @ORM\Column(nullable=true)
     * @var string
     */
    protected $youtubeUri;

    /**
     * @ORM\Column(nullable=true)
     * @var string
     */
    protected $sourceName;

    /**
     * @ORM\Column(nullable=true)
     * @var string
     */
    protected $authorName;

    /**
     * @ORM\Column(nullable=true)
     * @var string
     */
    protected $bookingUri;

    /**
     * Class constructor.
     *
     * This constructor is called before hydration and dependency
     * injection is happening.
     *
     * @return void
     */
    public function __construct()
    {
        $this->addresses = new ArrayCollection;
        $this->categories = new ArrayCollection;
        $this->files = new ArrayCollection;
        $this->media = new ArrayCollection;
    }

    /**
     * @return int
     */
    public function getMainType()
    {
        return $this->mainType;
    }

    /**
     * @param int $type
     * @return void
     */
    public function setMainType($type)
    {
        $this->mainType = $type;
    }

    /**
     * @return bool
     */
    public function getIsAttraction()
    {
        return $this->mainType == ArticleConstants::TYPE_ATTRACTION;
    }

    /**
     * @return bool
     */
    public function getIsGastronomy()
    {
        return $this->mainType == ArticleConstants::TYPE_GASTRONOMY;
    }

    /**
     * @return bool
     */
    public function getIsTour()
    {
        return $this->mainType == ArticleConstants::TYPE_TOUR;
    }

    /**
     * @return bool
     */
    public function getIsLodging()
    {
        return $this->mainType == ArticleConstants::TYPE_LODGING;
    }

    /**
     * @var string $name
     * @return void
     */
    public function setName($name)
    {
        $this->name = $name;
    }

    /**
     * @return string
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     * @var string $description
     * @return void
     */
    public function setDescription($description)
    {
        $this->description = $description;
    }

    /**
     * @return string
     */
    public function getDescription()
    {
        return $this->description;
    }

    /**
     * @var string $abstract
     * @return void
     */
    public function setAbstract($abstract)
    {
        $this->abstract = $abstract;
    }

    /**
     * @return string
     */
    public function getAbstract()
    {
        return $this->abstract;
    }

    /**
     * @var string $street
     * @return void
     */
    public function setStreet($street)
    {
        $this->street = $street;
    }

    /**
     * @return string
     */
    public function getStreet()
    {
        return $this->street;
    }

    /**
     * @var string $zip
     * @return void
     */
    public function setZip($zip)
    {
        $this->zip = $zip;
    }

    /**
     * @return string
     */
    public function getZip()
    {
        return $this->zip;
    }

    /**
     * @var string $city
     * @return void
     */
    public function setCity($city)
    {
        $this->city = $city;
    }

    /**
     * @return string
     */
    public function getCity()
    {
        return $this->city;
    }

    /**
     * @var string $country
     * @return void
     */
    public function setCountry($country)
    {
        $this->country = $country;
    }

    /**
     * @return string
     */
    public function getCountry()
    {
        return $this->country;
    }

    /**
     * Helper method to check if address fields are given.
     *
     * This only checks for zip code and city as e.g.
     * mountain huts won't have a street given.
     *
     * @return bool
     */
    public function getHasMainAddress()
    {
        return (!empty($this->zip) && !empty($this->city));
    }

    /**
     * @var float $latitude
     * @return void
     */
    public function setLatitude($latitude)
    {
        $this->latitude = $latitude;
    }

    /**
     * @return float
     */
    public function getLatitude()
    {
        return $this->latitude;
    }

    /**
     * @return bool
     */
    public function hasLatitude()
    {
        return !empty($this->latitude);
    }

    /**
     * @var float $longitude
     * @return void
     */
    public function setLongitude($longitude)
    {
        $this->longitude = $longitude;
    }

    /**
     * @return string
     */
    public function getLongitude()
    {
        return $this->longitude;
    }

    /**
     * @return bool
     */
    public function hasLongitude()
    {
        return !empty($this->longitude);
    }

    /**
     * @return bool
     */
    public function getHasGeocoordinates()
    {
        return ($this->hasLatitude() && $this->hasLongitude());
    }

    /**
     * @param Collection $categories
     * @return void
     */
    public function setCategories(Collection $categories)
    {
        $this->categories = $categories;
    }

    /**
     * @return Collection
     */
    public function getCategories()
    {
        return $this->categories;
    }

    /**
     * @param Collection $addresses
     * @return void
     */
    public function setAddresses(Collection $addresses)
    {
        $this->addresses = $addresses;
    }

    /**
     * @return Collection
     */
    public function getAddresses()
    {
        return $this->addresses;
    }

    /**
     * @param array $attributes
     * @return void
     */
    public function setAttributes(array $attributes)
    {
        $this->attributes = $attributes;
    }

    /**
     * @return array
     */
    public function getAttributes()
    {
        return $this->attributes;
    }

    /**
     * @param Collection $media
     * @return void
     */
    public function setMedia(Collection $media)
    {
        $this->media = $media;
    }

    /**
     * @return Collection
     */
    public function getMedia()
    {
        return $this->media;
    }

    /**
     * @var string|null $facebookUri
     * @return void
     */
    public function setFacebookUri($facebookUri)
    {
        $this->facebookUri = $facebookUri;
    }

    /**
     * @return string|null
     */
    public function getFacebookUri()
    {
        return $this->facebookUri;
    }

    /**
     * @var string|null $twitterUri
     * @return void
     */
    public function setTwitterUri($twitterUri)
    {
        $this->twitterUri = $twitterUri;
    }

    /**
     * @return string|null
     */
    public function getTwitterUri()
    {
        return $this->twitterUri;
    }

    /**
     * @var string|null $instagramUri
     * @return void
     */
    public function setInstagramUri($instagramUri)
    {
        $this->instagramUri = $instagramUri;
    }

    /**
     * @return string|null
     */
    public function getInstagramUri()
    {
        return $this->instagramUri;
    }

    /**
     * @var string|null $youtubeUri
     * @return void
     */
    public function setYoutubeUri($youtubeUri)
    {
        $this->youtubeUri = $youtubeUri;
    }

    /**
     * @return string|null
     */
    public function getYoutubeUri()
    {
        return $this->youtubeUri;
    }

    /**
     * @param Collection $files
     * @return void
     */
    public function setFiles(Collection $files)
    {
        $this->files = $files;
    }

    /**
     * @return Collection
     */
    public function getFiles()
    {
        return $this->files;
    }

    /**
     * @var string $sourceName
     * @return void
     */
    public function setSourceName($sourceName)
    {
        $this->sourceName = $sourceName;
    }

    /**
     * @return string
     */
    public function getSourceName()
    {
        return $this->sourceName;
    }

    /**
     * @var string|null $authorName
     * @return void
     */
    public function setAuthorName($authorName)
    {
        $this->authorName = $authorName;
    }

    /**
     * @return string|null
     */
    public function getAuthorName()
    {
        return $this->authorName;
    }

    /**
     * @var string|null $bookingUri
     * @return void
     */
    public function setBookingUri($bookingUri)
    {
        $this->bookingUri = $bookingUri;
    }

    /**
     * @return string|null
     */
    public function getBookingUri()
    {
        return $this->bookingUri;
    }

    public function jsonSerialize(): array
    {
        $category = null;
        if ($this->categories->count() > 0) {
            $category = $this->categories->first();
        }

        $tour = null;
        if (array_key_exists('geometry', $this->attributes) && $this->getIsTour()) {
            $tour = $this->attributes['geometry'];
        }

        $image = null;
        if ($this->media->count() > 0) {
            $image = [
                'url' => $this->media->first()->getSourceUri(),
                'alt' => $this->media->first()->getTitle()
            ];
        }


        return [
            'name' => $this->name,
            'lat' => $this->latitude,
            'lng' => $this->longitude,
            'category' => $category ? $category->getOriginalId() : null,
            'categoryTitle' => $category ? $category->getTitle() : null,
            'tour' => $tour,
            'image' => $image,
            'isTour' => $this->getIsTour(),
            'difficultyRating' => $this->attributes['difficultyRating'] ?? null,
            'outdoorActiveTrackingId' => $this->getOutdoorActiveId(),
            'url' => null,
        ];
    }

    private function getOutdoorActiveId()
    {
        if (strpos($this->originalId, 'outdooractive_') === false) {
            return null;
        }
        return str_replace('outdooractive_', '', $this->originalId);
    }
}
