<?php
namespace Newland\Toubiz\Sync\Neos\Importer;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Neos\Flow\Cli\CommandController;
use Newland\Toubiz\Api\ObjectAdapter\EventAdapterInterface;
use Newland\Toubiz\Sync\Neos\Domain\Model\Event;
use Newland\Toubiz\Sync\Neos\Domain\Repository\EventRepository;
use Newland\Toubiz\Sync\Neos\Service\ToubizService;

/**
 * Event importer.
 *
 * @Flow\Scope("singleton")
 */
class EventImporter extends AbstractImporter
{
    /**
     * @var EventAdapterInterface
     */
    protected $data;

    /**
     * @var Event
     */
    protected $event;

    /**
     * @Flow\Inject
     * @var EventRepository
     */
    protected $eventRepository;

    /**
     * Check if the given data should be imported.
     *
     * This is the case if no event exists in the database
     * or the data timestamp is newer than the one from the
     * already persisted event.
     *
     * @param EventAdapterInterface $data
     * @return bool
     */
    public function shouldBeImported(EventAdapterInterface $data)
    {
        $this->event = $this->eventRepository->findOneByOriginalId($data->getExternalId());
        return (!$this->event || $this->event->getUpdatedAt() < $data->getUpdatedAt());
    }

    /**
     * Import method.
     *
     * Persist given data by creating new objects or updating existing ones.
     *
     * @param EventAdapterInterface $data
     * @return Event
     */
    public function import($data)
    {
        // Check possibly previously fetched event if it matches the one from $data.
        if ($this->event && $this->event->getOriginalId() != $data->getExternalId()) {
            $this->event = $this->eventRepository->findOneByOriginalId($data->getExternalId());
        }
        $persisted = !!$this->event;
        $this->data = $data;

        if (!$this->event) {
            $this->event = new Event;
        }

        // Map simple values to have a valid event.
        $this->mapSimpleValues();

        // The event is being persisted in order to create relations to it.
        if ($persisted) {
            $this->eventRepository->update($this->event);
        } else {
            $this->eventRepository->add($this->event);
        }

        $this->mapAddresses();
        $this->mapCategories();
        $this->mapEventDates();
        $this->mapEventFeatures();
        $this->mapMedia();

        $this->eventRepository->update($this->event);

        // Persisting is required due to the API returning the same object multiple times!
        $this->persistenceManager->persistAll();
    }

    /**
     * Maps simple values onto the event.
     *
     * @return void
     */
    protected function mapSimpleValues()
    {
        if (empty($this->event->getOriginalId())) {
            $this->event->setOriginalId($this->data->getExternalId());
        }
        $this->event->setTitle($this->data->getName());
        $this->event->setDescription($this->data->getDescription());
        $this->event->setAdmission($this->data->getAdmission());
        $this->event->setIsHighlight($this->data->getIsHighlight());
        $this->event->setIsHighlight($this->data->getIsHighlight());
        $this->event->setBeginsAt($this->data->getBeginsAt());
        $this->event->setEndsAt($this->data->getEndsAt());
        $this->event->setUpdatedAt($this->data->getUpdatedAt());
    }

    /**
     * Imports and maps addresses.
     *
     * @return void
     */
    protected function mapAddresses()
    {
        $organizerItem = $this->data->getOrganizer();
        if ($organizerItem) {
            $importer = new AddressImporter;
            $importer->setParentObject($this->data);
            $organizer = $importer->import($organizerItem);
            $this->event->setOrganizer($organizer);
        } else {
            $this->event->setOrganizer(null);
        }

        $locationItem = $this->data->getOrganizer();
        if ($locationItem) {
            $importer = new AddressImporter;
            $importer->setParentObject($this->data);
            $location = $importer->import($locationItem);
            $this->event->setLocation($organizer);
        } else {
            $this->event->setLocation(null);
        }
    }

    /**
     * Maps categories onto the event.
     *
     * @return void
     */
    protected function mapCategories()
    {
        $categories = $this->event->getCategories();
        $categories->clear();
        foreach ($this->data->getCategories() as $data) {
            $importer = new CategoryImporter;
            $categories->add($importer->import($data));
        }
        $this->event->setCategories($categories);
    }

    /**
     * Maps event dates onto the event.
     *
     * @return void
     */
    public function mapEventDates()
    {
        $eventDates = $this->event->getEventDates();
        $eventDates->clear();
        foreach ($this->data->getEventDates() as $data) {
            $importer = new EventDateImporter;
            $importer->setEvent($this->event);
            $eventDates->add($importer->import($data));
        }
        $this->event->setEventDates($eventDates);
    }

    /**
     * Maps event features onto the event.
     *
     * @return void
     */
    public function mapEventFeatures()
    {
        $features = $this->event->getFeatures();
        $features->clear();
        foreach ($this->data->getEventFeatures() as $data) {
            $importer = new FeatureImporter;
            $features->add($importer->import($data));
        }
        $this->event->setFeatures($features);
    }

    /**
     * Maps multimedia events onto the event.
     *
     * @return void
     */
    public function mapMedia()
    {
        $media = $this->event->getMedia();
        $media->clear();
        foreach ($this->data->getMedia() as $data) {
            $importer = new MediumImporter;
            $importer->setEvent($this->event);
            $media->add($importer->import($data));
        }
        $this->event->setMedia($media);
    }
}
