<?php
namespace Newland\Toubiz\Sync\Neos\Domain\Repository;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Api\ObjectAdapter\Concern\TransportationFacilityConstants;
use Newland\Toubiz\Sync\Neos\Domain\Filter\TransportationFacilityFilter;
use Newland\Toubiz\Sync\Neos\Domain\Model\Region;

/**
 * Transportation facility repository.
 *
 * @Flow\Scope("singleton")
 */
class TransportationFacilityRepository extends AbstractRepository
{
    /**
     * Find facilities matching given filter.
     *
     * @param TransportationFacilityFilter $filter
     * @return QueryResultInterface
     */
    public function findByFilter(TransportationFacilityFilter $filter)
    {
        $query = $this->createQueryBuilder('transportationFacility');
        $this->applyBasicFilter($filter, $query);
        $query->leftJoin('transportationFacility.region', 'region');

        if ($filter->hasRegions()) {
            $query->andWhere(
                $query->expr()->in('transportationFacility.region', $filter->getRegions())
            );
        }
        if ($filter->hasType()) {
            $query->andWhere(
                $query->expr()->in('transportationFacility.type', $filter->getTypes())
            );
        }

        return $query->getQuery()->execute();
    }

    /**
     * Calculates statistics for region.
     *
     * @param Region|null $region
     * @return array
     */
    public function getStatistics(Region $region = null)
    {
        $query = $this->createQueryBuilder('transportationFacility');

        $constraint = '';
        if ($region) {
            $query->setParameter('regionId', $region->getPersistenceObjectIdentifier());
            $constraint = 'AND transportationFacility.region = :regionId';
        }

        $query->select('
            SUM (
                CASE WHEN
                    transportationFacility.type IN(
                        ' . TransportationFacilityConstants::TYPE_SKI_LIFT . ',
                        ' . TransportationFacilityConstants::TYPE_4_SEATER_CHAIRLIFT . ',
                        ' . TransportationFacilityConstants::TYPE_6_SEATER_CHAIRLIFT . ',
                        ' . TransportationFacilityConstants::TYPE_GONDOLA . '
                    )
                    AND transportationFacility.status = ' . TransportationFacilityConstants::STATUS_OPEN . '
                    ' . $constraint . '
                THEN 1 ELSE 0 END
            ) AS lifts_open,
            SUM (
                CASE WHEN
                    transportationFacility.type IN(
                        ' . TransportationFacilityConstants::TYPE_SKI_LIFT . ',
                        ' . TransportationFacilityConstants::TYPE_4_SEATER_CHAIRLIFT . ',
                        ' . TransportationFacilityConstants::TYPE_6_SEATER_CHAIRLIFT . ',
                        ' . TransportationFacilityConstants::TYPE_GONDOLA . '
                    )
                    ' . $constraint . '
                THEN 1 ELSE 0 END
            ) AS lifts_total,

            SUM (
                CASE WHEN
                    transportationFacility.type IN(
                        ' . TransportationFacilityConstants::TYPE_REVERSIBLE_AERIAL_TRAM . ',
                        ' . TransportationFacilityConstants::TYPE_COGWHEEL_RAILWAY . '
                    )
                    AND transportationFacility.status = ' . TransportationFacilityConstants::STATUS_OPEN . '
                    ' . $constraint . '
                THEN 1 ELSE 0 END
            ) AS mountain_railways_open,
            SUM (
                CASE WHEN
                    transportationFacility.type IN(
                        ' . TransportationFacilityConstants::TYPE_REVERSIBLE_AERIAL_TRAM . ',
                        ' . TransportationFacilityConstants::TYPE_COGWHEEL_RAILWAY . '
                    )
                    ' . $constraint . '
                THEN 1 ELSE 0 END
            ) AS mountain_railways_total
        ');
        return $query->getQuery()->execute()[0];
    }
}
