<?php
namespace Newland\Toubiz\Sync\Neos\Command;

/*
 * This file is part of the "toubiz-sync-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Api\Service\ServiceFactory;
use Newland\Toubiz\Sync\Neos\Importer\ArticleImporter;

/**
 * Articles command controller.
 *
 * Provides commands to manipulate article data.
 *
 * @Flow\Scope("singleton")
 */
class ArticlesCommandController extends AbstractCommandController
{
    /**
     * Synchronize command.
     *
     * Updates local articles database from API data source.
     *
     * @todo delete persisted articles not being in the API response
     * @return void
     */
    public function synchronizeCommand()
    {
        $configuration = $this->getConfigurationForService('Tportal/Api');
        if ($configuration) {
            $this->synchronizeLodgingsFromTportal($configuration);
        }

        $configuration = $this->getConfigurationForService('Toubiz/Legacy/Db');
        if ($configuration) {
            $this->synchronizePointOfInterestsFromDbService($configuration);
        }

        $configuration = $this->getConfigurationForService('Toubiz/Legacy/GastronomyApi');
        if ($configuration) {
            $this->synchronizeGastronomyFromApi($configuration);
        }

        $configuration = $this->getConfigurationForService('Outdooractive/Api');
        if ($configuration) {
            $this->synchronizeToursFromApi($configuration);
        }
    }

    /**
     * Synchronizes lodging data as articles from TPortal.
     *
     * @param array $configuration
     * @return void
     */
    protected function synchronizeLodgingsFromTportal($configuration)
    {
        $service = ServiceFactory::get('Tportal/Api');
        $service->setClientName($configuration['client']);

        $records = $service->fetchLodgings();
        if (!$records) {
            throw new \Exception('No lodging data received for synchronizing!');
        }

        foreach ($records as $record) {
            $importer = new ArticleImporter;
            $importer->import($record);
        }
    }

    /**
     * Synchronizes POI data from legacy toubiz DB service.
     *
     * @param array $configuration
     * @return void
     */
    protected function synchronizePointOfInterestsFromDbService($configuration)
    {
        $service = ServiceFactory::get('Toubiz/Legacy/Db');
        $service->setClientName($configuration['client']);
        $service->setApiKey($configuration['apiKey']);

        $records = $service->fetchPointOfInterests();
        if (!$records) {
            throw new \Exception('No POI data received for synchronizing!');
        }

        foreach ($records as $record) {
            $importer = new ArticleImporter;
            $importer->import($record);
        }
    }

    /**
     * Synchronizes gastronomy data from legacy API.
     *
     * @param array $configuration
     * @return void
     */
    protected function synchronizeGastronomyFromApi($configuration)
    {
        $service = ServiceFactory::get('Toubiz/Legacy/GastronomyApi');
        $service->setApiKey($configuration['apiKey']);

        $records = $service->fetchRecords();
        if (!$records) {
            throw new \Exception('No gastronomy data received for synchronizing!');
        }

        foreach ($records as $record) {
            $importer = new ArticleImporter;
            $importer->import($record);
        }
    }

    /**
     * Synchronizes tours.
     *
     * @param array $configuration
     * @return void
     */
    protected function synchronizeToursFromApi($configuration)
    {
        $service = ServiceFactory::get('Outdooractive/Api');
        $service->setClientName($configuration['client']);
        $service->setApiKey($configuration['apiKey']);

        $records = $service->fetchTours();
        if (!$records) {
            throw new \Exception('No tour data received for synchronizing!');
        }

        foreach ($records as $record) {
            $importer = new ArticleImporter;
            $importer->import($record);
        }
    }
}
