<?php
namespace Newland\Toubiz\SnowReport\Neos\Controller;

/*
 * This file is part of the "toubiz-snowreport-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Neos\Flow\Mvc\Controller\ActionController;
use Newland\Toubiz\Sync\Neos\Domain\Filter\RegionFilter;
use Newland\Toubiz\Sync\Neos\Domain\Filter\SlopeFilter;
use Newland\Toubiz\Sync\Neos\Domain\Filter\TransportationFacilityFilter;
use Newland\Toubiz\Sync\Neos\Domain\Model\Region;
use Newland\Toubiz\Sync\Neos\Domain\Model\Slope;
use Newland\Toubiz\Sync\Neos\Domain\Repository\RegionRepository;
use Newland\Toubiz\Sync\Neos\Domain\Repository\SlopeRepository;
use Newland\Toubiz\Sync\Neos\Domain\Repository\TransportationFacilityRepository;

/**
 * Regions controller.
 *
 * Responsible for handling region data.
 *
 * @Flow\Scope("singleton")
 */
class RegionsController extends ActionController
{
    /**
     * @Flow\Inject
     * @var RegionRepository
     */
    protected $regionRepository;

    /**
     * @Flow\Inject
     * @var SlopeRepository
     */
    protected $slopeRepository;

    /**
     * @Flow\Inject
     * @var TransportationFacilityRepository
     */
    protected $transportationFacilityRepository;

    /**
     * Index action.
     *
     * Displays an overview of regions.
     *
     * @return void
     */
    public function indexAction()
    {
        $regionFilter = new RegionFilter;
        $regionFilter->setOrderBy(['region.name' => 'asc']);

        $regions = [];
        foreach ($this->regionRepository->findByFilter($regionFilter) as $region) {
            $regions[] = [
                'region' => $region,
                'statistics' => [
                    'facilities' => $this->transportationFacilityRepository->getStatistics([$region]),
                    'slopes' => $this->slopeRepository->getStatistics([$region])
                ]
            ];
        }
        $this->view->assign('regions', $regions);
    }

    /**
     * Show action.
     *
     * Shows details about a given region.
     *
     * @param Region $region
     * @return void
     */
    public function showAction(Region $region)
    {
        $this->view->assignMultiple([
            'region' => $region,
            'statistics' => [
                'facilities' => $this->transportationFacilityRepository->getStatistics([$region]),
                'slopes' => $this->slopeRepository->getStatistics([$region])
            ],
            'transportationFacilities' => $this->transportationFacilityRepository->findByRegion($region)
        ]);

        $filter = new TransportationFacilityFilter;
        $filter->setRegionCollection([$region]);
        $filter->setOrderBy(['transportationFacility.name' => 'asc']);
        $this->view->assign(
            'transportationFacilities',
            $this->transportationFacilityRepository->findByFilter($filter)
        );

        $filter = new SlopeFilter;
        $filter->setRegionCollection([$region]);
        $filter->setOrderBy(['slope.name' => 'asc']);

        $filter->setTypes([Slope::TYPE_SKI]);
        $this->view->assign('ski', $this->slopeRepository->findByFilter($filter));

        $filter->setTypes([Slope::TYPE_SLED_TRAIL]);
        $this->view->assign('sledding', $this->slopeRepository->findByFilter($filter));

        $filter->setTypes([Slope::TYPE_HIKING_TRAIL]);
        $this->view->assign('hiking', $this->slopeRepository->findByFilter($filter));

        $filter->setTypes([Slope::TYPE_SNOWSHOE_HIKING_TRAIL]);
        $this->view->assign('snowshoeHiking', $this->slopeRepository->findByFilter($filter));

        $filter->setTypes([Slope::TYPE_CROSS_COUNTRY_SKIING, Slope::TYPE_CROSS_COUNTRY_SKIING_SKATING]);
        $this->view->assign('crossCountrySkiing', $this->slopeRepository->findByFilter($filter));
    }
}
