<?php declare(strict_types=1);
namespace Newland\Toubiz\Search\Neos\Command;

/*
 * This file is part of the "toubiz-search-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Neos\Flow\Cli\CommandController;
use Neos\Flow\Configuration\Exception\InvalidConfigurationException;
use Neos\Flow\ObjectManagement\Exception\InvalidObjectException;
use Newland\Contracts\Neos\Search\IndexerInterface;
use Newland\Contracts\Neos\Search\SearchBackend;
use Newland\Contracts\Neos\Search\ProgressHandler;
use Newland\Toubiz\Search\Neos\Backend\SearchBackendFactory;
use Newland\Toubiz\Search\Neos\Progress\NullHandler;
use Newland\Toubiz\Search\Neos\Progress\ProgressBarHandler;
use Symfony\Component\Console\Helper\ProgressBar;

/**
 * Indexer command controller.
 *
 * @Flow\Scope("singleton")
 */
class IndexerCommandController extends CommandController
{
    /**
     * @Flow\InjectConfiguration
     * @var array
     */
    protected $configuration;

    /** @var SearchBackend */
    protected $searchBackend;
    public function injectSearchBackend(SearchBackendFactory $factory): void
    {
        $this->searchBackend = $factory->getSearchBackendFromConfiguration();
    }

    /**
     * Updates the index based on registered indexers.
     *
     * @param bool $quiet
     * @param string $only
     * @return void
     */
    public function runCommand(bool $quiet = false, string $only = ''): void
    {
        $progressHandler = $quiet ? new NullHandler() : new ProgressBarHandler($this->output->getOutput());

        foreach ($this->configuration['indexers'] as $className => $configuration) {
            if (!$configuration['enable']) {
                continue;
            }

            $this->searchBackend->setSource($className);
            $this->runIndexer($className, $configuration, $progressHandler, $this->searchBackend);
        }
    }

    public function clearCommand(): void
    {
        foreach ($this->configuration['indexers'] as $className => $configuration) {
            $this->output->outputLine($className);
            $this->searchBackend->setSource($className);
            $this->searchBackend->deleteObsoleteIndexEntries([ ]);
        }
    }

    private function runIndexer(
        string $className,
        array $configuration,
        ProgressHandler $progressHandler,
        SearchBackend $backend
    ): void {
        $indexer = $this->getIndexer($className, $configuration);

        $indexer->index($backend, $progressHandler);
        $backend->afterIndexing();
        $indexer->postIndex($backend);

        $this->outputLine('');
    }


    private function getIndexer(string $className, array $configuration): IndexerInterface
    {
        $indexer = new $className;
        if (!($indexer instanceof IndexerInterface)) {
            throw new InvalidObjectException(
                'Indexers must implement ' . IndexerInterface::class . ' - ' . $className . ' does not.'
            );
        }
        if (array_key_exists('configuration', $configuration)) {
            $indexer->setConfiguration($configuration['configuration']);
        }
        return $indexer;
    }
}
