<?php
namespace Newland\Toubiz\Search\Neos\Controller;

/*
 * This file is part of the "toubiz-search-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Neos\Flow\Mvc\Controller\ActionController;
use Neos\ContentRepository\Domain\Model\Node;
use Newland\Toubiz\Search\Neos\Domain\Model\SearchQuery;
use Newland\Toubiz\Search\Neos\Domain\Model\SearchResult;
use Newland\Toubiz\Search\Neos\Domain\Model\History\SearchQuery as HistorySearchQuery;
use Newland\Toubiz\Search\Neos\Domain\Repository\SearchIndexRepository;
use Newland\Toubiz\Search\Neos\Domain\Repository\History\SearchQueryRepository as HistorySearchQueryRepository;
use Newland\Toubiz\Search\Neos\ObjectFinder\ObjectFinderFactory;

/**
 * Searches controller.
 *
 * @Flow\Scope("singleton")
 */
class SearchesController extends ActionController
{
    /**
     * @var HistorySearchQueryRepository
     * @Flow\Inject()
     */
    protected $historySearchQueryRepository;

    /**
     * @var SearchIndexRepository
     * @Flow\Inject()
     */
    protected $searchIndexRepository;

    /**
     * @var ObjectFinderFactory
     * @Flow\Inject()
     */
    protected $objectFinderFactory;

    /**
     * Result action.
     *
     * Renders search results for the given search parameters.
     *
     * @param SearchQuery $searchQuery
     * @param integer $page
     * @return void
     */
    public function resultAction(SearchQuery $searchQuery = null, $page = 1): void
    {
        /** @var Node $node */
        $node = $this->request->getInternalArgument('__node');
        $searchQuery = $searchQuery ?? SearchQuery::scopedForSiteContainingNode($node);
        $searchResults = $this->executeSearchQuery($searchQuery, $page);

        $this->view->assignMultiple(
            [
                'searchQuery' => $searchQuery,
                'node' => $node,
                'pagination' => $this->buildPaginationData($searchQuery, $page),
                'searchResults' => $searchResults,
            ]
        );
    }

    /**
     * Action called via AJAX to return search results.
     *
     * This does not store historical records as it is mainly used for pagination.
     *
     * @param SearchQuery $searchQuery
     * @param Node $node
     * @param int $page
     * @return void
     */
    public function ajaxResultAction(SearchQuery $searchQuery = null, Node $node = null, $page = null): void
    {
        if (!$searchQuery) {
            $searchQuery = $this->buildSearchQueryFromPluginArguments();
        }

        $searchIndexes = $this->searchIndexRepository->search($searchQuery, $page ?? 1);
        $this->view->assignMultiple(
            [
                'node' => $node,
                'searchQuery' => $searchQuery,
                'searchResults' => $this->buildSearchResults($searchIndexes),
                'pagination' => $this->buildPaginationData($searchQuery, $page),
            ]
        );
    }

    /**
     * Stores search query in history.
     *
     * @param SearchQuery $searchQuery
     * @return void
     */
    protected function storeHistoricalQuery(SearchQuery $searchQuery): void
    {
        $history = new HistorySearchQuery;
        $history->setQueryString($searchQuery->getQueryString());
        $history->setScopes($searchQuery->getScopes());
        $this->historySearchQueryRepository->add($history);
    }

    /**
     * Converts given search index instances to search results.
     *
     * @param array $searchIndexes
     * @return array
     */
    protected function buildSearchResults(array $searchIndexes): array
    {
        $results = [];
        foreach ($searchIndexes as $index) {
            $result = new SearchResult;
            $result->setSearchIndex($index);

            $finder = $this->objectFinderFactory->get($index->getSource());
            $finder->setSearchIndex($index);
            $result->setOriginalObject($finder->find());

            $results[] = $result;
        }
        return $results;
    }

    /**
     * Through page frame provider, the object can't be instantiated
     * automatically due to namespace differences in the argument names.
     *
     * @return SearchQuery
     */
    protected function buildSearchQueryFromPluginArguments(): SearchQuery
    {
        $arguments = $this->request->getParentRequest()->getArguments();
        $searchQuery = new SearchQuery;
        if (array_key_exists('--plugin', $arguments)) {
            $searchQuery->setArguments($arguments['--plugin']['searchQuery']);
        }
        return $searchQuery;
    }

    protected function buildPaginationData(SearchQuery $searchQuery, $page): ?array
    {
        if ($searchQuery->isEmpty()) {
            return null;
        }

        $count = $this->searchIndexRepository->countSearch($searchQuery);
        return [
            'isFirst' => $page === 1,
            'isLast' => $count['pages'] <= $page,
            'count' => $count,
            'page' => $page,
        ];
    }


    private function executeSearchQuery(SearchQuery $searchQuery, int $page): array
    {
        if ($searchQuery->isEmpty()) {
            return [];
        }
        $this->storeHistoricalQuery($searchQuery);
        $searchIndexes = $this->searchIndexRepository->search($searchQuery, $page);
        return $this->buildSearchResults($searchIndexes);
    }
}
