<?php declare(strict_types=1);
namespace Newland\Toubiz\Search\Neos\Indexer;

/*
 * This file is part of the "toubiz-search-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\ContentRepository\Domain\Model\Node;
use Neos\ContentRepository\Domain\Service\Context;
use Neos\ContentRepository\Domain\Service\ContextFactoryInterface;
use Neos\Eel\InterpretedEvaluator;
use Neos\Flow\Annotations as Flow;
use Neos\Neos\Domain\Service\ContentContext;

/**
 * @Flow\Scope("singleton")
 */
abstract class AbstractNodeIndexer
{
    /**
     * Neos content dimension configuration.
     *
     * @Flow\InjectConfiguration(
     *   package="Neos.ContentRepository",
     *   path="contentDimensions"
     * )
     * @var array
     */
    protected $contentDimensions;

    /**
     * @var ContextFactoryInterface
     * @Flow\Inject()
     */
    protected $contextFactory;

    /**
     * @var InterpretedEvaluator
     * @Flow\Inject()
     */
    protected $eelEvaluator;

    /** @var array */
    protected $configuration;

    public function setConfiguration(array $configuration = []): void
    {
        $this->configuration = $configuration;
    }

    /**
     * Builds a context for querying with flow query.
     *
     * @param array $options
     * @return ContentContext
     */
    protected function getContext(array $options = []): ContentContext
    {
        /** @var ContentContext $context */
        $context = $this->contextFactory->create(
            array_merge(
                [
                    'workspaceName' => 'live',
                    'currentDateTime' => new \Neos\Flow\Utility\Now(),
                    'dimensions' => [],
                    'invisibleContentShown' => false,
                    'removedContentShown' => false,
                    'inaccessibleContentShown' => false,
                ],
                $options
            )
        );

        return $context;
    }

    /**
     * Checks if a given node should be indexed.
     *
     * @param Node $node
     * @return bool
     */
    protected function nodeShouldBeIndexed(Node $node): bool
    {
        if ($node->getProperty('excludeFromSearch')) {
            return false;
        }

        if ($node->isHidden()) {
            return false;
        }

        if (!$this->nodeTypeShouldBeIndexed($node)) {
            return false;
        }

        /** @var \DateTime|null $hiddenBefore */
        $hiddenBefore = $node->getHiddenBeforeDateTime();
        if ($hiddenBefore && $hiddenBefore > new \DateTime()) {
            return false;
        }
        /** @var \DateTime|null $hiddenAfter */
        $hiddenAfter = $node->getHiddenBeforeDateTime();
        if ($hiddenAfter && $hiddenAfter < new \DateTime()) {
            return false;
        }

        return true;
    }

    /**
     * Checks if a given node type should be indexed.
     *
     * @param Node $node
     * @return bool
     */
    protected function nodeTypeShouldBeIndexed(Node $node): bool
    {
        foreach ($this->configuration['nodeTypes'] as $nodeType => $configuration) {
            if (array_key_exists('index', $configuration) && $node->getNodeType()->isOfType($nodeType)) {
                return $configuration['index'];
            }
        }

        // Index in in case of no configuration telling the opposite.
        return true;
    }

    /**
     * Strips ignored strings from index.
     *
     * @param Node $node
     * @param string $value
     * @return string
     */
    protected function stripIgnoredStrings(Node $node, string $value): string
    {
        foreach ($this->configuration['nodeTypes'] as $nodeType => $configuration) {
            if (!$node->getNodeType()->isOfType($nodeType)) {
                continue;
            }

            if (!array_key_exists('ignoredStrings', $configuration)) {
                continue;
            }

            foreach ($configuration['ignoredStrings'] as $string) {
                $value = str_ireplace($string, '', $value);
            }
        }

        return $value;
    }

    /**
     * Evaluates configured title eel expression for given node.
     *
     * @param Node $node
     * @return string|null
     */
    protected function getTitleForNode(Node $node): ?string
    {
        foreach ($this->configuration['nodeTypes'] as $nodeType => $configuration) {
            if (!$node->getNodeType()->isOfType($nodeType)) {
                continue;
            }

            if (!empty($configuration['title'])) {
                return \Neos\Eel\Utility::evaluateEelExpression(
                    $configuration['title'],
                    $this->eelEvaluator,
                    ['node' => $node]
                );
            }
        }

        return $node->getProperty('title');
    }

    /**
     * Evaluates configured description eel expression for given node.
     *
     * @param Node $node
     * @return string|null
     */
    protected function getDescriptionForNode(Node $node): ?string
    {
        foreach ($this->configuration['nodeTypes'] as $nodeType => $configuration) {
            if (!$node->getNodeType()->isOfType($nodeType)) {
                continue;
            }

            if (!empty($configuration['description'])) {
                $description = \Neos\Eel\Utility::evaluateEelExpression(
                    $configuration['description'],
                    $this->eelEvaluator,
                    ['node' => $node]
                );

                if ($description) {
                    return $description;
                }
            }
        }

        return null;
    }

    /**
     * Gets the scope for the given node.
     *
     * @param Node $node
     * @return string
     */
    protected function getScopeForNode(Node $node): ?string
    {
        if (!preg_match('/^(\/sites\/.*?)\//', $node->getPath(), $matches)) {
            return null;
        }

        return $matches[1] ?? null;
    }

    /**
     * Gets the language dimension for this node.
     *
     * @param Node $node
     * @return string|null
     */
    protected function getLanguageForNode(Node $node): ?string
    {
        if (array_key_exists('language', $node->getDimensions())) {
            return $node->getDimensions()['language'][0];
        }
        return null;
    }

    /**
     * Returns configured content dimensions.
     *
     * @return array
     */
    protected function getContentDimensions(): array
    {
        return $this->contentDimensions;
    }
}
