<?php

declare(strict_types=1);

namespace Newland\Toubiz\Search\Neos\Eel\Helper;

use Neos\Utility\Arrays;
use Neos\Eel\ProtectedContextAwareInterface;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Configuration\ConfigurationManager;

use Newland\Toubiz\Search\Neos\Backend\ElasticSearchConnector;

/**
 * Eel Helper to process the ElasticSearch Query Result
 */
class SearchResultHelper implements ProtectedContextAwareInterface
{
    /**
     * @Flow\Inject()
     * @var ElasticSearchConnector
     */
    protected $connector;

    /**
     * @var ConfigurationManager
     * @Flow\Inject()
     */
    protected $configurationManager;

    /**
     * Run search query using ElasticSearchConnector.
     *
     * Returns hits object of Elastic Search API Response Content.
     *
     * @param string $searchTerm
     * @param int $itemsPerPage
     * @param string $currentPage
     * @param string $language
     * @param string $site
     * @param array $userEnabledScopes
     */
    public function query(
        string $searchTerm,
        int $itemsPerPage,
        string $currentPage,
        string $language,
        string $site,
        array $userEnabledScopes
    ) {
        $currentPage = (int) $currentPage;
        $queryIndexFrom = ($currentPage - 1) * $itemsPerPage;
        $resultSize = $itemsPerPage;
        $simpleQueryStringQuery = [
            'fields' => ['title^3', 'description^2', 'content'],
        ];

        // spaces exist
        if (strpos($searchTerm, ' ') !== false) {
            $simpleQueryStringQuery['query'] = trim($searchTerm);
        } else if (strpos($searchTerm, '*') === false) {
            $simpleQueryStringQuery['query'] = $searchTerm;
        } else {
            $simpleQueryStringQuery['query'] = $searchTerm . '*';
            $simpleQueryStringQuery['flags'] = 'PREFIX';
        }

        $boolQuery = [
            'must' => [
                // Match search term with prefix operator
                [
                    'simple_query_string' => $simpleQueryStringQuery,
                ],

                // Match language
                [
                    'match' => [
                        'language' => $language,
                    ],
                ],
            ],
        ];

        $shouldMatchScopes = [];

        $availableScopes = $this->configurationManager->getConfiguration(
            ConfigurationManager::CONFIGURATION_TYPE_SETTINGS,
            'Newland.Toubiz.Search.Neos.availableScopes'
        );

        $activeScopes = [];

        // Set enabled scopes based on user enabled ones or settings
        if (!empty($userEnabledScopes)) {
            $activeScopes = $userEnabledScopes;
        } else {
            foreach ($availableScopes as $scope => $isEnabled) {
                if ($isEnabled) {
                    $activeScopes[] = $scope;
                }
            }
        }

        // Add enabled scopes
        foreach ($activeScopes as $scope) {
            // Add each specified scope as match query.
            array_push($shouldMatchScopes, [
                'match' => [
                    'scope' => $scope === 'page' ? $site : $scope,
                ],
            ]);
        }

        // Set match queries as should block.
        $boolQuery['should'] = $shouldMatchScopes;

        // Match at least one scope
        $boolQuery['minimum_should_match'] = 1;

        $searchResponse = $this->connector->search([
            'sort' => ['_score'],
            'from' => $queryIndexFrom,
            'size' => $resultSize,
            'query' => [
                'bool' => $boolQuery,
            ],
        ]);

        $content = $searchResponse->getTreatedContent();
        return Arrays::getValueByPath($content, 'hits');
    }

    /**
     * @param string $methodName
     * @return boolean
     */
    public function allowsCallOfMethod($methodName)
    {
        return true;
    }
}
