<?php declare(strict_types=1);
namespace Newland\Toubiz\Search\Neos\Command;

/*
 * This file is part of the "toubiz-search-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Neos\Flow\Cli\CommandController;
use Neos\Flow\Configuration\Exception\InvalidConfigurationException;
use Neos\Flow\ObjectManagement\Exception\InvalidObjectException;
use Newland\Contracts\Neos\Search\IndexerInterface;
use Newland\Contracts\Neos\Search\IndexingBackend;
use Newland\Contracts\Neos\Search\ProgressHandler;
use Newland\Toubiz\Search\Neos\Progress\NullHandler;
use Newland\Toubiz\Search\Neos\Progress\ProgressBarHandler;
use Symfony\Component\Console\Helper\ProgressBar;

/**
 * Indexer command controller.
 *
 * @Flow\Scope("singleton")
 */
class IndexerCommandController extends CommandController
{
    /**
     * The configuration for this package.
     *
     * @Flow\InjectConfiguration
     * @var array
     */
    protected $configuration;

    /**
     * Updates the index based on registered indexers.
     *
     * @param bool $quiet
     * @param string $only
     * @return void
     */
    public function runCommand(bool $quiet = false, string $only = ''): void
    {
        $progressHandler = $quiet ? new NullHandler() : new ProgressBarHandler($this->output->getOutput());
        $backend = $this->backend();

        foreach ($this->configuration['indexers'] as $className => $configuration) {
            if (!$configuration['enable']) {
                continue;
            }

            $backend->setSource($className);
            $this->runIndexer($className, $configuration, $progressHandler, $backend);
        }
    }

    public function clearCommand(): void
    {
        $backend = $this->backend();
        foreach ($this->configuration['indexers'] as $className => $configuration) {
            $this->output->outputLine($className);
            $backend->setSource($className);
            $backend->deleteObsoleteIndexEntries([ ]);
        }
    }

    private function runIndexer(
        string $className,
        array $configuration,
        ProgressHandler $progressHandler,
        IndexingBackend $backend
    ): void {
        $indexer = $this->getIndexer($className, $configuration);

        $indexer->index($backend, $progressHandler);
        $indexer->postIndex($backend);

        $this->outputLine('');
    }


    private function getIndexer(string $className, array $configuration): IndexerInterface
    {
        $indexer = new $className;
        if (!($indexer instanceof IndexerInterface)) {
            throw new InvalidObjectException(
                'Indexers must implement ' . IndexerInterface::class . ' - ' . $className . ' does not.'
            );
        }
        if (array_key_exists('configuration', $configuration)) {
            $indexer->setConfiguration($configuration['configuration']);
        }
        return $indexer;
    }

    private function backend(): IndexingBackend
    {
        $className = $this->configuration['backend'] ?? null;

        if (empty($className) || !class_exists($className)) {
            throw new InvalidConfigurationException('
                Please be sure to set `Newland.Toubiz.Search.Neos.backend`
                to a valid fully qualified classname
            ');
        }

        $instance = $this->objectManager->get($className);
        if (!($instance instanceof IndexingBackend)) {
            throw new InvalidObjectException(sprintf(
                'The class configured in `Newland.Toubiz.Search.Neos.backend` must implement %s but %s does not.',
                IndexingBackend::class,
                $className
            ));
        }

        return $instance;
    }
}
