<?php declare(strict_types=1);

namespace Newland\Toubiz\Search\Neos\Backend;

use Neos\Flow\Annotations as Flow;

use Newland\Contracts\Neos\Search\IndexingBackend;
use Newland\Contracts\Neos\Search\IndexRecordModification;

use Neos\Utility\Arrays;
use function Safe\preg_replace;

class ElasticSearchBackend implements IndexingBackend
{
    /** @var string */
    protected $source;

    /**
     * @Flow\Inject()
     * @var ElasticSearchConnector
     */
    protected $connector;

    public function setSource(string $source): void
    {
        // extract last part from source string to set source name
        $sourceStringArray = explode('\\', $source);
        $this->source = end($sourceStringArray);
    }

    /**
     * Sets index object content and creates/updates it in the index
     * via ElasticSearchConnector.
     *
     * @param IndexRecordModification $modification
     **/
    public function createOrUpdateIndexEntry(IndexRecordModification $modification): void
    {
        $identifier = $modification->getIdentifier() . '-' . $modification->getLanguage();

        // Set index object content.
        $indexContent = [
            'content' => $this->convertToIndexableString($modification->getContent()),
            'title' => $this->convertToIndexableString($modification->getTitle()),
            'description' => $this->convertToIndexableString($modification->getDescription()),
            'scope' => $modification->getScope(),
            'language' => $modification->getLanguage(),
            'identifier' => $modification->getIdentifier(),
            'source' => $this->source
        ];

        // Pass index to ElasticSearchConnector.
        $this->connector->addOrUpdateObject($indexContent, $identifier);
    }

    /**
     * Delete all index entries but the ones in $identifiersToRetain.
     *
     * @param string[] $identifiersToRetain
     **/
    public function deleteObsoleteIndexEntries(array $identifiersToRetain): void
    {
        if (!empty($identifiersToRetain)) {
            // Only get all objects of current source context.
            $sourceContextSearchResponse = $this->connector->search([
                'size' => 10000,
                'query' => [
                    'bool' => [
                        'must' => [
                            'match' => [
                                'source' => $this->source
                            ]
                        ]
                    ]
                ]
            ]);

            // Extract content and result hits.
            $content = $sourceContextSearchResponse->getTreatedContent();
            $hits = Arrays::getValueByPath($content, 'hits.hits');

            // Remove object if it is not found in $identifiersToRetain
            foreach ($hits as $object) {
                if (!in_array($object['_source']['identifier'], $identifiersToRetain, true)) {
                    $this->connector->removeObject($object['_id']);
                }
            }
        }
    }

    protected function convertToIndexableString(string $input): string
    {
        // Remove HTML tags.
        $string = strip_tags($input);

        // Replace html entities like `&nbsp;`.
        $string = html_entity_decode($string);

        // Replace newlines with a space to shorten the string.
        $string = preg_replace('/[\r\n]+/', ' ', $string);

        if (\is_array($string)) {
            $string = implode(' ', $string);
        }

        return $string;
    }
}
