<?php declare(strict_types=1);

namespace Newland\Toubiz\Search\Neos\Backend;

use Neos\Flow\Persistence\PersistenceManagerInterface;
use Newland\Contracts\Neos\Search\IndexingBackend;
use Newland\Contracts\Neos\Search\IndexRecordModification;
use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Search\Neos\Domain\Model\SearchIndex;
use Newland\Toubiz\Search\Neos\Domain\Repository\SearchIndexRepository;
use function Safe\preg_replace;

class DatabaseRecordBackend implements IndexingBackend
{

    /**
     * @var SearchIndexRepository
     * @Flow\Inject()
     */
    protected $searchIndexRepository;

    /**
     * @var PersistenceManagerInterface
     * @Flow\Inject()
     */
    protected $persistenceManager;

    /** @var int */
    protected $persistEvery = 100;

    /** @var int */
    protected $indexed = 0;

    /** @var string */
    protected $source;

    public function setSource(string $source): void
    {
        $this->source = $source;
    }

    public function createOrUpdateIndexEntry(IndexRecordModification $modification): void
    {
        $this->searchIndexRepository->modifyItem(
            $modification->getIdentifier(),
            $this->source,
            function (SearchIndex $index) use ($modification) {
                $index->setIdentifier($modification->getIdentifier());
                $index->setContent($this->convertToIndexableString($modification->getContent()));
                $index->setTitle($this->convertToIndexableString($modification->getTitle()));
                $index->setDescription($modification->getDescription());
                $index->setScope($modification->getScope());
                $index->setLanguage($modification->getLanguage());
                $index->setSource($this->source);
            }
        );

        $this->persistenceManager->persistAll();
        if (++$this->indexed % $this->persistEvery === 0) {
            $this->persistenceManager->clearState();
        }
    }

    /** @param string[] $identifiersToRetain */
    public function deleteObsoleteIndexEntries(array $identifiersToRetain): void
    {
        $this->searchIndexRepository->deleteObsolete($identifiersToRetain, $this->source);
    }

    protected function convertToIndexableString(string $input): string
    {
        // Remove HTML tags.
        $string = strip_tags($input);

        // Replace html entities like `&nbsp;`.
        $string = html_entity_decode($string);

        // Replace newlines with a space to shorten the string.
        $string = preg_replace('/[\r\n]+/', ' ', $string);

        if (\is_array($string)) {
            return implode(' ', $string);
        }

        return $string;
    }
}
