<?php
namespace Newland\Toubiz\Search\Neos\Domain\Repository;

/*
 * This file is part of the "toubiz-search-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Doctrine\ORM\QueryBuilder;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Persistence\Doctrine\Repository;
use Newland\Toubiz\Search\Neos\Domain\Model\SearchQuery;
use Newland\Toubiz\Search\Neos\Domain\Search\SearchInterface;

/**
 * Search index repository.
 *
 * @Flow\Scope("singleton")
 */
class SearchIndexRepository extends Repository
{
    /**
     * @var integer
     */
    const LIMIT = 10;

    /**
     * The configuration for this package.
     *
     * @Flow\InjectConfiguration(path="search")
     * @var array
     */
    protected $configuration;

    /**
     * Finds one record with the given identifier and source.
     *
     * @param string $identifier
     * @param string $source
     * @return mixed
     */
    public function findOneByIdentifierAndSource($identifier, $source)
    {
        $query = $this->createQuery();
        $query->matching(
            $query->logicalAnd(
                $query->equals('identifier', $identifier),
                $query->equals('source', $source)
            )
        );
        $query->setLimit(1);
        return $query->execute()->getFirst();
    }

    /**
     * Search method.
     *
     * Executes a search for the given search query with the configured
     * search implementation.
     *
     * @param SearchQuery $searchQuery
     * @param int $page
     * @return array
     */
    public function search(SearchQuery $searchQuery, $page)
    {
        $queryBuilder = $this->getSearchClass()->getQuery($searchQuery, $this);
        $this->addConstraints($queryBuilder, $searchQuery);

        $query = $queryBuilder->getQuery();
        $query->setMaxResults(self::LIMIT)
              ->setFirstResult(($page - 1) * self::LIMIT);

        return $query->execute();
    }

    /**
     * Search count method.
     *
     * Counts the search results.
     *
     * @param SearchQuery $searchQuery
     * @return array
     */
    public function countSearch(SearchQuery $searchQuery)
    {
        $queryBuilder = $this->getSearchClass()->getQuery($searchQuery, $this);
        $this->addConstraints($queryBuilder, $searchQuery);
        $total = $queryBuilder->select('COUNT(i)')->getQuery()->getSingleScalarResult();
        $pages = floor($total / self::LIMIT) + ($total % self::LIMIT == 0 ? 0 : 1);
        return [
            'total' => $total,
            'pages' => ($pages == 0 ? 1 : $pages),
            'limit' => self::LIMIT
        ];
    }

    /**
     * Drops all records from given source that do not have an identifier
     * that is mentioned in $activeIdentifiers.
     *
     * @param array $activeIdentifiers
     * @param string $source
     * @return void
     */
    public function deleteObsolete(array $activeIdentifiers, $source)
    {
        if (empty($activeIdentifiers)) {
            return;
        }

        $query = $this->createQueryBuilder('i');
        $query->delete()
              ->where('i.source = :source')
              ->andWhere($query->expr()->notIn('i.identifier', $activeIdentifiers))
              ->setParameter('source', $source);

        $query->getQuery()->execute();
    }

    /**
     * Instantiates the configured search class.
     *
     * @return SearchInterface
     */
    protected function getSearchClass()
    {
        return new $this->configuration['class']($this->configuration['configuration']);
    }

    /**
     * Adds common constraints to the query builder.
     *
     * @param QueryBuilder &$queryBuilder
     * @param SearchQuery $searchQuery
     * @return void
     */
    protected function addConstraints(QueryBuilder &$queryBuilder, SearchQuery $searchQuery)
    {
        $queryBuilder->andWhere('(i.scope = :scope OR i.scope IS NULL)')
                     ->setParameter('scope', $searchQuery->getScope());

        if ($searchQuery->hasLanguage()) {
            if ($this->configuration['configuration']['languageHandling']['strict']) {
                $queryBuilder->andWhere('(i.language = :language AND i.language IS NOT NULL)')
                             ->setParameter('language', $searchQuery->getLanguage());
            } else {
                $queryBuilder->andWhere('(i.language = :language OR i.language IS NULL)')
                             ->setParameter('language', $searchQuery->getLanguage());
            }
        }
    }
}
