# Toubiz search package for Neos CMS.

This package provides search functionality into Neos.


## Integration
Add the package to your project's `composer.json` and, to ensure proper loading order of configurations, also to your site package's `composer.json`.

To build/update the index, run `php flow indexer:run`. On server environments, you'd want to add a crob job line that executes `FLOW_CONTEXT=Production php flow indexer run`.

## Configuration

### Search relevance
The default relevance is set to `5` next to the provided `0` for `lowest` and `10` for `highest`. If the project requires a more fine-grained relevance tuning, you can add additional values in between the existing ones.

In the example below, `3` and `7` are added:
```
'Neos.Neos:Document':
  properties:
    searchRelevance:
      editorOptions:
        values:
          3:
            label: Low
          7:
            label: High
```


#### DocumentNodeIndexer
The document node indexer indexes all properties from a document node and its content node children. By default, boolean and numerical values are excluded from the index.

The indexer follows a blacklist-approach. This way, editors don't have to be reminded to manually add all of their properties of content nodes to a search configuration. However, sometimes it makes sense to exclude certain properties from the index as they might not bring useful or even wrong information (from the viewpoint of a search index). To ignore certain properties, you simply add them to the list.
```
Newland:
  Toubiz:
    Search:
      Neos:
        indexers:
          'Newland\Toubiz\Search\Neos\Indexer\DocumentNodeIndexer':
            configuration:
              nodeTypes:
                'Neos.Neos:Document':
                  ignoredProperties:
                    - myCustomIgnoredProperty
                    - anotherProperty
```

Some properties may contain default values that are hidden or not overwritten by the editor. To strip default values from the search index, configure them accordingly.
```
Newland:
  Toubiz:
    Search:
      Neos:
        indexers:
          'Newland\Toubiz\Search\Neos\Indexer\DocumentNodeIndexer':
            configuration:
              nodeTypes:
                'Neos.Neos:Document':
                  ignoredStrings:
                    - Pre-Line
                    - Title goes here
```

For each indexed entry, a title and description is stored for use in search results. You can override both with custom flow queries to adapt them to your needs.
```
Newland:
  Toubiz:
    Search:
      Neos:
        indexers:
          'Newland\Toubiz\Search\Neos\Indexer\DocumentNodeIndexer':
            configuration:
              title: ${q(node).property('title')}
              description: ${q(node).children('main').find('[instanceof Neos.Neos:Content][content != ""]').property('content')}
```

## Custom search form outside the plugin
To integrate a custom search form outside of a plugin scope (e.g. in a page header), you simply need to build a form that passes the data the same as the plugin would. Be sure to utilize the SearchActionUri view helper as it provides the correct URI based on the current language dimension.
```
{namespace search=Newland\Toubiz\Search\Neos\ViewHelpers}

<f:form package="Newland.Toubiz.Search.Neos"
        controller="Searches"
        action="results"
        actionUri="{search:searchActionUri()}"
        name="searchQuery"
        fieldNamePrefix="--newland_toubiz_search_neos-search"
        objectName="searchQuery">

    <f:form.textfield property="queryString" />
    <f:form.hidden property="scope" value="{site.path}" />
    <f:form.hidden property="language" value="{node.dimensions.language.0}" />
</f:form>
```

Attention: If you're using the page frame provider, replace `fieldNamePrefix="--newland_toubiz_search_neos-search"` with `--plugin`!

The `queryString` property is the actual search query string. The `scope` defines which site has to be searched; If you want to search all sites, you can leave it blank. The `language` property obviously constrains search results to the currently selected language (or records without a language set).


## Implementing own indexers and result renderings.
To index custom records and their rendering, you need a few things.

### The indexer
Obviously, an indexer class is needed which fills the search index with your data. This is quite simple and you can have a look at the `IndexerInterface` as well as the `AbstractIndexer` class. Depending on what you want to implement, you may also have a look at the `AbstractNodeIndexer` or the `AbstractRecordIndexer`.

Your custom indexers can be configured inside your project's or package's `Settings.yaml`.
```
Newland:
  Toubiz:
    Search:
      Neos:
        indexers:
          'Vendor\PackageName\Indexers\MyCustomIndexer':
            enable: true
            configuration: []
```

When you now run `php flow indexer:run`, the command controller will pick up your class and execute the expected methods.

### The object finder
While the indexer is only storing main data into the database table, an object finder is required to access the original object from your search result in the search results view in order to customize appearance with data from the original object (which is missing in the search index). For this, have a look at the `AbstractObjectFinder` and the `ObjectFinderInterface` as well as the `DocumentNodeObjectFinder` for an example.

Your object finder is configured next to your indexer inside the `configuration` scope:
```
# ...
indexers:
  'Vendor\PackageName\Indexers\MyCustomIndexer':
    enable: true
    configuration:
      objectFinder: 'Vendor\PackageName\ObjectFinder\MyCustomObjectFinder'
```
