<?php
namespace Newland\Toubiz\Search\Neos\Indexer;

/*
 * This file is part of the "toubiz-search-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\ContentRepository\Domain\Model\Node;
use Neos\Eel\FlowQuery\FlowQuery;
use Neos\Flow\Annotations as Flow;
use Neos\Media\Domain\Repository\AssetRepository;
use Newland\Toubiz\Search\Neos\Extractor\ExtractorFactory;

/**
 * Asset indexer.
 *
 * Indexes assets. This indexer is based on the node indexer,
 * as only assets that are in use by public nodes should be indexed.
 *
 * @Flow\Scope("singleton")
 */
class AssetIndexer extends AbstractNodeIndexer implements IndexerInterface
{
    /**
     * @var AssetRepository
     * @Flow\Inject
     */
    protected $assetRepository;

    /**
     * Entry point.
     *
     * Gets all content dimensions to index assets for.
     *
     * @return void
     */
    public function index()
    {
        $dimensions = $this->getContentDimensions();
        if (empty($dimensions)) {
            $this->indexContentDimension([]);
            return;
        }

        foreach ($dimensions as $dimensionName => $dimension) {
            foreach ($dimension['presets'] as $preset) {
                $this->indexContentDimension(
                    [
                        $dimensionName => $preset['values']
                    ]
                );
            }
        }
    }

    public function postIndex()
    {
        // TODO
    }

    /**
     * Indexes document nodes from given content dimension.
     *
     * @param array $dimensions
     * @return void
     */
    protected function indexContentDimension($dimensions = [])
    {
        $context = $this->getContext(['dimensions' => $dimensions]);
        $documentNodes = (new FlowQuery([$context->getCurrentSiteNode()]))
            ->children('[instanceof Neos.Neos:Document]');

        foreach ($documentNodes as $node) {
            $this->indexAssetsFromDocumentNode($node);
        }
    }

    /**
     * Indexes assets from a document node.
     *
     * @param Node $documentNode
     * @return void
     */
    protected function indexAssetsFromDocumentNode(Node $documentNode)
    {
        $childNodes = (new FlowQuery([$documentNode]))->find('[instanceof Neos.Neos:Content]');
        foreach ($childNodes as $childNode) {
            $this->indexAssetsFromContentNode($childNode);
        }
    }

    /**
     * Indexes assets from a content node.
     *
     * @param Node $node
     * @return void
     */
    protected function indexAssetsFromContentNode(Node $node)
    {
        foreach ($node->getProperties() as $name => $value) {
            if (!is_string($value)) {
                continue;
            }

            if (!preg_match_all('/(asset:\/\/[\w]{8}-[\w]{4}-[\w]{4}-[\w]{4}-[\w]{12})/i', $value, $matches, PREG_SET_ORDER)) {
                continue;
            }

            foreach ($matches[0] as $assetUri) {
                $this->indexAssetUri($assetUri);
            }
        }
    }

    /**
     * Indexes an asset by the given asset:// URI.
     *
     * @param string $uri
     * @return void
     */
    protected function indexAssetUri($uri)
    {
        $identifier = substr($uri, 8);
        $asset = $this->assetRepository->findByIdentifier($identifier);
        if (!$asset) {
            return;
        }

        // TODO get type of asset
        // label should be $asset->getLabel()

        $resource = $asset->getResource();
        $extractor = ExtractorFactory::getByMimeType($resource->getMediaType());
        if (!$extractor) {
            return;
        }

        $extractor->setResource($resource);
        $index = [
            'name' => $asset->getLabel(),
            'content' => $extractor->extract()
        ];

        // TODO add to index
        var_dump($index);
    }
}
