<?php
namespace Newland\Toubiz\Search\Neos\Indexer;

/*
 * This file is part of the "toubiz-search-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Search\Neos\Domain\Model\SearchIndex;
use Newland\Toubiz\Search\Neos\Domain\Repository\SearchIndexRepository;

/**
 * Abstract indexer.
 *
 * @Flow\Scope("singleton")
 */
abstract class AbstractIndexer
{
    /**
     * @var array Indexer configuration.
     */
    protected $configuration = [];

    /**
     * @var array The index for this instance.
     */
    protected $index = [];

    /**
     * @var SearchIndexRepository
     * @Flow\Inject
     */
    protected $searchIndexRepository;

    /**
     * Sets configurations for the current indexer.
     *
     * @var array $configuration
     * @return void
     */
    public function setConfiguration(array $configuration = [])
    {
        $this->configuration = $configuration;
    }

    /**
     * Persists the given data array to the index.
     *
     * @param array $data
     * @return void
     */
    protected function addToIndex(array $data)
    {
        $this->index[] = $data['identifier'];
        $this->persistIndexRecord($data);
    }

    /**
     * Convert the given string to one that is suitable for searching
     * and/or representing.
     *
     * @param string $input
     * @return string
     */
    protected function convertToIndexableString($input)
    {
        // Remove HTML tags.
        $string = strip_tags($input);

        // Replace html entities like `&nbsp;`.
        $string = html_entity_decode($string);

        // Replace newlines with a space to shorten the string.
        $string = preg_replace('/[\r\n]+/', ' ', $string);

        return (string) $string;
    }

    /**
     * Persists index data for given node.
     *
     * @param array $indexData
     * @return void
     */
    protected function persistIndexRecord($indexData)
    {
        $record = $this->searchIndexRepository->findOneByIdentifierAndSource(
            $indexData['identifier'],
            get_class($this)
        );

        if (!$record) {
            $persisted = false;
            $record = new SearchIndex;
            $record->setSource(get_class($this));
            $record->setIdentifier($indexData['identifier']);
        } else {
            $persisted = true;
        }

        $record->setContent($indexData['content']);
        $record->setTitle($indexData['title']);
        $record->setDescription($indexData['description']);

        if (array_key_exists('scope', $indexData)) {
            $record->setScope($indexData['scope']);
        }
        if (array_key_exists('language', $indexData)) {
            $record->setLanguage($indexData['language']);
        }

        if ($persisted) {
            $this->searchIndexRepository->update($record);
        } else {
            $this->searchIndexRepository->add($record);
        }
    }
}
