<?php
namespace Newland\Toubiz\Search\Neos\Controller;

/*
 * This file is part of the "toubiz-search-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Neos\Flow\Mvc\Controller\ActionController;
use Neos\ContentRepository\Domain\Model\Node;
use Newland\Toubiz\Search\Neos\Domain\Model\SearchQuery;
use Newland\Toubiz\Search\Neos\Domain\Model\SearchResult;
use Newland\Toubiz\Search\Neos\Domain\Model\History\SearchQuery as HistorySearchQuery;
use Newland\Toubiz\Search\Neos\Domain\Repository\SearchIndexRepository;
use Newland\Toubiz\Search\Neos\Domain\Repository\History\SearchQueryRepository as HistorySearchQueryRepository;
use Newland\Toubiz\Search\Neos\ObjectFinder\ObjectFinderFactory;

/**
 * Searches controller.
 *
 * @Flow\Scope("singleton")
 */
class SearchesController extends ActionController
{
    /**
     * @var HistorySearchQueryRepository
     * @Flow\Inject
     */
    protected $historySearchQueryRepository;

    /**
     * @var SearchIndexRepository
     * @Flow\Inject
     */
    protected $searchIndexRepository;

    /**
     * @var ObjectFinderFactory
     * @Flow\Inject
     */
    protected $objectFinderFactory;

    /**
     * Result action.
     *
     * Renders search results for the given search parameters.
     *
     * @param SearchQuery $searchQuery
     * @param integer $page
     * @return void
     */
    public function resultAction(SearchQuery $searchQuery = null, $page = 1)
    {
        if ($searchQuery && !$searchQuery->isEmpty()) {
            $this->storeHistoricalQuery($searchQuery);
            $searchIndexes = $this->searchIndexRepository->search($searchQuery, $page);

            $this->view->assignMultiple([
                'searchResults' => $this->buildSearchResults($searchIndexes),
                'searchResultsCount' => $this->searchIndexRepository->countSearch($searchQuery)
            ]);
        } else {
            $searchQuery = new SearchQuery;
        }

        $node = $this->request->getInternalArgument('__node');
        $this->view->assignMultiple([
            'searchQuery' => $searchQuery,
            'node' => $node,
            'scope' => $this->extractScopeFromNode($node)
        ]);
    }

    /**
     * Stores search query in history.
     *
     * @param SearchQuery $searchQuery
     * @return void
     */
    protected function storeHistoricalQuery(SearchQuery $searchQuery)
    {
        $history = new HistorySearchQuery;
        $history->setQueryString($searchQuery->getQueryString());
        $history->setScope($searchQuery->getScope());
        $this->historySearchQueryRepository->add($history);
    }

    /**
     * Converts given search index instances to search results.
     *
     * @param array $searchIndex
     * @return array
     */
    protected function buildSearchResults(array $searchIndexes)
    {
        $results = [];
        foreach ($searchIndexes as $index) {
            $result = new SearchResult;
            $result->setSearchIndex($index);

            $finder = $this->objectFinderFactory->get($index->getSource());
            $finder->setSearchIndex($index);
            $result->setOriginalObject($finder->find());

            $results[] = $result;
        }
        return $results;
    }

    /**
     * Extracts the scope (site node path) from the given node.
     *
     * @param Node $node
     * @return string
     */
    protected function extractScopeFromNode(Node $node)
    {
        $path = explode('/', $node->getPath());
        return '/' . $path[1] . '/' . $path[2];
    }
}
