<?php declare(strict_types=1);
namespace Newland\Toubiz\Search\Neos\ViewHelpers;

/*
 * This file is part of the "toubiz-search-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\Flow\Mvc\ActionRequest;
use Neos\FluidAdaptor\Core\ViewHelper\AbstractViewHelper;
use Neos\Flow\Annotations as Flow;
use function Safe\preg_match;

/**
 * Search action URI view helper.
 *
 * Returns the correct action URI for the search page.
 *
 * @Flow\Scope("singleton")
 */
class SearchActionUriViewHelper extends AbstractViewHelper
{
    protected const FALLBACK_LANGUAGE = 'en';

    /**
     * The configuration for this package.
     *
     * @Flow\InjectConfiguration(path="uriPathSegments")
     * @var array
     */
    protected $uriPathSegments;

    /**
     * Neos language dimension configuration.
     *
     * @Flow\InjectConfiguration(
     *   package="Neos.ContentRepository",
     *   path="contentDimensions.language"
     * )
     * @var array
     */
    protected $languageDimension;

    /**
     * @var \Neos\ContentRepository\Domain\Repository\NodeDataRepository
     * @Flow\Inject
     */
    protected $nodeDataRepository;

    public function render(): string
    {
        $language = $this->getLanguageDimension();
        if (!array_key_exists($language, $this->uriPathSegments)) {
            return '/search';
        }

        $pathSegments = [$this->uriPathSegments[$language]];
        $segment = $this->getUriSegmentForLanguage();
        if (isset($segment)) {
            array_unshift($pathSegments, $segment);
        }

        return '/' . implode('/', array_filter($pathSegments));
    }

    /**
     * Gets the configured uri segment for the current language.
     *
     * @return string|null
     */
    protected function getUriSegmentForLanguage(): ?string
    {
        $language = $this->getLanguageDimension();
        if (array_key_exists('presets', $this->languageDimension)
            && array_key_exists($language, $this->languageDimension['presets'])
        ) {
            return $this->languageDimension['presets'][$language]['uriSegment'];
        }
        return null;
    }

    /**
     * Gets the current language dimension string.
     *
     * If no internal `node` argument is given (which is a Node object),
     * the internal node request argument (which is a context path string) is
     * used and the language is fetched from it directly.
     *
     * @return string
     */
    protected function getLanguageDimension(): string
    {
        $node = $this->node();
        if ($node) {
            $dimensions = $node->getDimensions();
            if (array_key_exists('language', $dimensions)) {
                return $dimensions['language'][0];
            }
        }

        preg_match('/language=(.+)/', $this->nodePath(), $matches);
        return $matches[1] ?? static::FALLBACK_LANGUAGE;
    }

    private function node(): ?NodeInterface
    {
        $request = $this->controllerContext->getRequest();
        if (!($request instanceof ActionRequest)) {
            return null;
        }

        /** @var NodeInterface $node */
        $node = $request->getInternalArgument('__node');
        return $node;
    }

    private function nodePath(): string
    {
        $request = $this->controllerContext->getRequest();
        if (!($request instanceof ActionRequest)) {
            return '';
        }

        if (!$request->hasArgument('node')) {
            return '';
        }

        return $request->getArgument('node') ?? '';
    }
}
