<?php declare(strict_types=1);

namespace Newland\Toubiz\Search\Neos\Utility;

use Neos\ContentRepository\Domain\Model\NodeType;
use Neos\Flow\Annotations as Flow;

/**
 * @Flow\Scope("singleton")
 */
class NodeTypeUtility
{
    /**
     * @var array<string, array<string, int>>
     */
    private $distanceCache = [];

    /**
     * Returns an associative array that uses the node type as a key and the depth / distance to
     * the node type as a value. This array will contain every super type of the given node type.
     *
     * If node type A has a super type B which has a super type C, then the distance between A and
     * C is 2 (since 2 'hops' are required to get from A to C).
     *
     * @return int[]
     */
    public function getParentNodeTypeDepths(NodeType $startingPoint): array
    {
        if (array_key_exists($startingPoint->getName(), $this->distanceCache)) {
            return $this->distanceCache[$startingPoint->getName()];
        }

        $distances = [];
        $queue = [ [ 0, $startingPoint ] ];

        // phpcs:ignore Generic.CodeAnalysis.AssignmentInCondition.FoundInWhileCondition
        while ([ $depth, $nodeType ] = array_shift($queue)) {
            /** @var NodeType $nodeType */

            // Only consider closest distance.
            if (array_key_exists($nodeType->getName(), $distances)) {
                continue;
            }

            $distances[$nodeType->getName()] = (int) $depth;
            foreach ($nodeType->getDeclaredSuperTypes() as $superType) {
                $queue[] = [ $depth + 1, $superType ];
            }
        }

        $this->distanceCache[$startingPoint->getName()] = $distances;
        return $distances;
    }

    /**
     * Takes an associative configuration array that uses node types as a key and arbitrary values
     * and returns an array of the same format that fulfils the 2 following conditions:
     *
     * 1. Only contains super types of the given node type (and the node type itself)
     * 2. Is sorted by applicability (more specific node type configurations are sorted up)
     */
    public function sortAndFilterConfigurationByNodeType(
        NodeType $nodeType,
        array $configuration
    ): array {
        $onlyApplicable = array_filter(
            $configuration,
            function (string $nodeTypeName) use ($nodeType) {
                return $nodeType->isOfType($nodeTypeName);
            },
            ARRAY_FILTER_USE_KEY
        );


        $depths = $this->getParentNodeTypeDepths($nodeType);
        uksort(
            $onlyApplicable,
            function (string $a, string $b) use ($depths) {
                return ($depths[$a] ?? 99) - ($depths[$b] ?? 99);
            }
        );

        return $onlyApplicable;
    }
}
