<?php declare(strict_types=1);

namespace Newland\Toubiz\Search\Neos\Backend\ElasticSearch;

use Flowpack\ElasticSearch\Domain\Factory\ClientFactory;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Utility\Environment;
use function Safe\preg_replace;

class ObjectTypeFactory
{

    /**
     * @var string|null
     * @Flow\InjectConfiguration(package="Newland.Toubiz.Search.Neos", path="elastic.index")
     */
    protected $configuredIndexName;

    /**
     * @var array|null
     * @Flow\InjectConfiguration(package="Neos.Flow", path="persistence.backendOptions")
     */
    protected $databaseSettings;

    /**
     * @Flow\Inject
     * @var ClientFactory
     */
    protected $clientFactory;

    /**
     * @var Environment
     * @Flow\Inject()
     */
    protected $environment;

    public function getType(): ObjectType
    {
        $indexName = $this->getIndexName();
        $client = $this->clientFactory->create();
        $index = $client->findIndex($indexName);
        return new ObjectType($index);
    }

    /**
     * Returns the configured index name if exists or a index name based on the hostname & environment if not.
     */
    protected function getIndexName(): string
    {
        if ($this->configuredIndexName) {
            return $this->configuredIndexName;
        }

        $context = 'unknown';
        if ($this->environment->getContext()->isDevelopment()) {
            $context = 'development';
        } elseif ($this->environment->getContext()->isTesting()) {
            $context = 'testing';
        } elseif ($this->environment->getContext()->isProduction()) {
            $context = 'production';
        }

        $name = implode('__', [
            gethostname(),
            $this->databaseSettings['user'],
            $this->databaseSettings['dbname'],
            $context,
        ]);

        return $this->normalizeName($name);
    }

    private function normalizeName(string $name): string
    {
        $name = strtolower($name);
        $name = preg_replace('/[^\w\-_]/', '-', $name);
        if (\is_array($name)) {
            return implode('-', $name);
        }
        return $name;
    }
}
