# Toubiz search package for Neos CMS.

This package provides search functionality into Neos. It uses elasticsearch to index and retrieve results.

## Setup
### Prerequisites
- You need a project running neos 4.x.
- ElasticSearch: You need access to an elastic search server in order to setup the search package.

### 1. Install

Add the search package to your project's composer.json and, to ensure proper loading order of configurations, also to your site package's `composer.json`.

If it was included before, run:
```
$ composer update newland/toubiz-search-neos
```

It is probably a good idea to update the other packages from neos-toubiz-frontend as well because they share versions and are usually updated together.

**(warning) Migration notes**: 

If a previous version of toubiz-search was installed, you can now safely remove this part from `./Configuration/Routes.yaml`:

```
- name: 'Search'
  uriPattern: '<SearchSubroutes>'
  subRoutes:
    'SearchSubroutes':
      package: 'Newland.Toubiz.Search.Neos'
```

as well as this part from `./Configuration/Views.yaml` (could look slightly different depending on the project):
```
- requestFilter: 'isPackage("Neos.ContentRepository.Search") && isSubPackage("ViewHelpers\Widget") && isController("Paginate") && isAction("index")'
  options:
    templatePathAndFilename: 'resource://Newland.BaseTheme/Private/Templates/Search/Paginate.html'
```

### 2. Configure

Configure the elastic search client and index: This works in a similar way as the database configuration which is different on local, preview and live. Thus, these settings will be added individually for each instance:

(local) `./Configuration/Development/Settings.yaml`

```
Flowpack:
  ElasticSearch:
    # Host of the elastic search server where data is indexed and retrieved from.
    # user and pw are empty, thats correct. Access is handled over ip
    clients:
      default:
        -
          host: 'mc-13.lis-cms.de'
          port: 9200
          username: ''
          password: ''
 
# Project-specific name of the index where data is indexed and retrieved from.
Newland:
  Toubiz:
    Search:
      Neos:
        elastic:
          index: 'index-001-test'
 ```

OR: 

(live) `./DMT-001/live/shared/Configuration/Production/Settings.yaml`

```
Flowpack:
  ElasticSearch:
    # Host of the elastic search server where data is indexed and retrieved from.
    # user and pw are empty, thats correct. Access is handled over ip
    clients:
      default:
        -
          host: 'localhost'
          port: 9200
          username: ''
          password: ''
 
 
# If no project specific index name is defined, then a guaranteed unique one is generated based on the project configuration.
```

Username and password is usually not required because access is controlled via IP whitelisting. 

On live instance, localhost is used because elastic search should run on the same server as the project.


**(info) Optional**:
Configure overrides for available scopes. For example if not all implemented scopes should be enabled (See the default configuration of the search package to check possible values):

`Settings.Toubiz.Search.Neos.yaml`

```
Newland:
  Toubiz:
    Search:
      Neos:
        availableScopes:
          'offer': false
          'package': false
```

In this example, we disable offer and package scopes. See the default config of the search package to see all implemented scopes.

### 3. Build Index

To build the index, run:
```
$ php flow indexer:run
```

On production environments, you'd want to add a cron job line that executes it like this:
```
FLOW_CONTEXT=Production php flow indexer run
```

This should run every time a sync happened or individual articles/pages are removed.

### 4. Add styling

Import the styles for the search components in your project: 

```
@import '((INSERTRELATIVEPATHTO))/Plugins/Newland.Toubiz.Search.Neos/Resources/Private/Styles/index';
```

The path is relative to the current location where you make the import. Don't be confused if other product style imports exist which get the styles from the toubiz-styleguide. The search package is the first package which follows newer best practice and brings its own styles.

**(warning) Migration note**: 
If migrating from an older search version, adjust search style overrides if exiting. Also make sure you remove old imports:

```
@import 'node_modules/@nimius/toubiz-styleguide/products/search/index'; // REMOVE
```
 
**(warning) Purge Note**:
If after importing and rebuilding your stylesheet the styles still do not appear, make sure the purge config whitelist includes the plugin template and partial files, e.g:
```
'./Packages/Plugins/**/*.html',
```

(Path is relative to location of gulpfile.js)

***(info) Optional***:
Add style overrides/theming if needed, e.g. in styles subfolder `/toubiz/overrides/_search.scss`

### 5. (Optional) Add custom search form

Go through these steps if the project uses a global search form, e.g. in the header.

#### 1. Set search result page reference

`Settings.Toubiz.Search.Neos.yaml`

```
Newland:
  Toubiz:
    Search:
      Neos:
        # node reference to search page
        mainSearchPage: 'node://6bd95a34-d81a-4d78-a761-f0c0c08b20f0'
```

This node reference will be used in the form action. You can retrieve the node id of the page from the neos inspector settings tab while the page node is selected in the content tree. Expand the section with additional information to see it.

**(warning) Note**:
Be careful as the node id of the search page might be different in live/preview/local environment.

Also make sure to assign it in fusion, e.g: 

Page.fusion (in the body section) to have it available in the page partial:
```
searchPage = ${Configuration.setting('Newland.Toubiz.Search.Neos.mainSearchPage')}
```

#### Include the markup for the form

The included input is required to have the name `search` as this is the name of the parameter required by the search package.

`PATH/TO/SEARCHFORM/PARTIAL.html`

```
{namespace neos=Neos\Neos\ViewHelpers}
 
<f:if condition="{searchPage}">
    <form action="{neos:uri.node(node: searchPage)}">
        <input
            id="search-input"
            name="search"
            class="c-search__input"
            placeholder="Ihre Suche" />
 
        <button
            type="submit"
            class="c-button c-button--ghost"
            aria-labelledby="search-label">
 
            <svg class="c-icon m-0" aria-hidden="true" focusable="false">
                <use xlink:href="#icon-search" />
            </svg>
        </button>
    </form>
</f:if>
```

## Editorial Integration
Go to any page you want the search form and results to appear and insert the `Search` plugin:

![](./Resources/Documentation/node-list.png)

#### Inspector Settings

Configure the following settings on the plugin node:

- **Items per page**:  Defines how many items should be displayed per page. Default is 10. Only set if you want a different number.
- **Preselected scopes**: A select box which is empty by default, indicating all configured available scopes are enabled. If individual scopes are added to the select box, only those will be enabled for this search plugin. ![](./Resources/Documentation/types.png). Scope "Pages"/"Seiten" never includes pages from other microsites should they exist. 

## Configuration of page indexing

#### DocumentNodeIndexer
The document node indexer indexes all properties from a document node and its content node children. By default, boolean and numerical values are excluded from the index.

The indexer follows a blacklist-approach. This way, editors don't have to be reminded to manually add all of their properties of content nodes to a search configuration. However, sometimes it makes sense to exclude certain properties from the index as they might not bring useful or even wrong information (from the viewpoint of a search index). To ignore certain properties, you simply add them to the list.
```
Newland:
  Toubiz:
    Search:
      Neos:
        indexers:
          'Newland\Toubiz\Search\Neos\Indexer\DocumentNodeIndexer':
            configuration:
              nodeTypes:
                'Neos.Neos:Document':
                  ignoredProperties:
                    - myCustomIgnoredProperty
                    - anotherProperty
```

Some properties may contain default values that are hidden or not overwritten by the editor. To strip default values from the search index, configure them accordingly.
```
Newland:
  Toubiz:
    Search:
      Neos:
        indexers:
          'Newland\Toubiz\Search\Neos\Indexer\DocumentNodeIndexer':
            configuration:
              nodeTypes:
                'Neos.Neos:Document':
                  ignoredStrings:
                    - Pre-Line
                    - Title goes here
```

For each indexed entry, a title and description is stored for use in search results. You can override both with custom flow queries to adapt them to your needs.
```
Newland:
  Toubiz:
    Search:
      Neos:
        indexers:
          'Newland\Toubiz\Search\Neos\Indexer\DocumentNodeIndexer':
            configuration:
              title: ${q(node).property('title')}
              description: ${q(node).children('main').find('[instanceof Neos.Neos:Content][content != ""]').property('content')}
```

## Implementing own indexers and result renderings.
To index custom records and their rendering, you need a few things.

### The indexer
Obviously, an indexer class is needed which fills the search index with your data. This is quite simple and you can have a look at the `IndexerInterface` as well as the `AbstractIndexer` class. Depending on what you want to implement, you may also have a look at the `AbstractNodeIndexer` or the `AbstractRecordIndexer`.

Your custom indexers can be configured inside your project's or package's `Settings.yaml`.
```
Newland:
  Toubiz:
    Search:
      Neos:
        indexers:
          'Vendor\PackageName\Indexers\MyCustomIndexer':
            enable: true
            configuration: []
```

When you now run `php flow indexer:run`, the command controller will pick up your class and execute the expected methods.

### Rendering
In order to add custom rendering for certain results, add cases to the `Newland.Toubiz.Search.Neos:SearchResultRenderer` prototype.

```
prototype(Newland.Toubiz.Search.Neos:SearchResultRenderer).foo {
      condition = ${searchHit._source.source == 'Vendor\Namespace\MyCustomIndexer'}
      renderer = Newland.Toubiz.Search.Neos:GenericSearchResult {
          url = 'https://foobar.com'
          title = ${'custom title: ' + searchHit._source.title}
      }
}
```

See also:
* [Newland.Toubiz.Search.Neos:SearchResultRenderer](./Resources/Private/Fusion/Search/SearchResultRenderer.fusion)
* [Newland.Toubiz.Search.Neos:GenericSearchResult](./Resources/Private/Fusion/Search/GenericSearchResult.fusion)
