<?php declare(strict_types=1);
namespace Newland\Toubiz\Search\Neos\Domain\Model;

/*
 * This file is part of the "toubiz-search-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Doctrine\ORM\Mapping as ORM;
use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\Flow\Annotations as Flow;

/**
 * Search query model.
 *
 * Represents a search query.
 *
 * @Flow\Scope("singleton")
 */
class SearchQuery
{

    public static function scopedForSiteContainingNode(NodeInterface $node): SearchQuery
    {
        $searchQuery = new SearchQuery();
        $path = explode('/', $node->getPath());
        $searchQuery->setScopes([ '/' . $path[1] . '/' . $path[2] ]);
        return $searchQuery;
    }

    /**
     * The query string.
     *
     * @var string
     */
    protected $queryString = '';

    /**
     * The scopes to be searched.
     *
     * @var string[]|null
     * @ORM\Column(nullable=TRUE, type="array")
     */
    protected $scopes;

    /**
     * The language to be searched.
     *
     * @var string|null
     * @ORM\Column(nullable=TRUE)
     */
    protected $language;

    /**
     * Sets given arguments.
     *
     * @param array $arguments
     * @return void
     */
    public function setArguments(array $arguments): void
    {
        foreach ($arguments as $name => $value) {
            $this->$name = $value;
        }
    }

    public function setQueryString(string $queryString): void
    {
        $this->queryString = $queryString;
    }

    public function getQueryString(): string
    {
        return $this->queryString;
    }

    public function getSearchWords(): array
    {
        return explode(' ', $this->queryString);
    }

    public function isEmpty(): bool
    {
        return preg_match('/\w+/', $this->queryString) === false;
    }

    /**
     * @param string[]|null $scopes
     * @return void
     */
    public function setScopes(?array $scopes): void
    {
        $this->scopes = $scopes;
    }

    /**
     * @return string[]|null
     */
    public function getScopes(): ?array
    {
        return $this->scopes;
    }

    public function setLanguage(?string $language): void
    {
        $this->language = $language;
    }

    public function getLanguage(): ?string
    {
        return $this->language;
    }

    public function hasLanguage(): bool
    {
        return !empty($this->language);
    }

    /**
     * Setter for setting a singular scope.
     * This is done to support both, supplying a single scope using a form field with the `scope`
     * property and supplying multiple ones using `scopes.{N}`.
     *
     * @param string $scope
     */
    public function setScope(string $scope): void
    {
        $this->scopes = [ $scope ];
    }
}
