<?php declare(strict_types=1);
namespace Newland\Toubiz\Search\Neos\Command;

/*
 * This file is part of the "toubiz-search-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Neos\Flow\Cli\CommandController;
use Neos\Flow\ObjectManagement\Exception\InvalidObjectException;
use Newland\Toubiz\Search\Neos\Indexer\IndexerInterface;
use Symfony\Component\Console\Helper\ProgressBar;

/**
 * Indexer command controller.
 *
 * @Flow\Scope("singleton")
 */
class IndexerCommandController extends CommandController
{
    /**
     * The configuration for this package.
     *
     * @Flow\InjectConfiguration
     * @var array
     */
    protected $configuration;

    /**
     * Run command.
     *
     * Updates the index based on registered indexers.
     *
     * @param bool $quiet
     * @return void
     */
    public function runCommand(bool $quiet = false)
    {
        foreach ($this->configuration['indexers'] as $className => $configuration) {
            if (!$configuration['enable']) {
                continue;
            }

            $this->runIndexer($className, $configuration, $quiet);
        }
    }

    private function runIndexer(string $className, array $configuration, bool $quiet): void
    {
        $indexer = $this->getIndexer($className, $configuration);

        if (!$quiet) {
            $this->addProgressBar($indexer);
        }

        $indexer->index();
        $indexer->postIndex();

        $this->outputLine('');
    }

    private function addProgressBar(IndexerInterface $indexer): void
    {
        $progress = new ProgressBar($this->output->getOutput());
        $progress->setFormat('%current% [%bar%] ' . $indexer->name());
        $progress->start(0);
        $indexer->onProgress(function (int $processed, ?int $max) use ($progress) {
            if ($max !== null) {
                $progress->setMaxSteps($max);
            }
            $progress->setProgress($processed);
        });
    }

    private function getIndexer(string $className, array $configuration): IndexerInterface
    {
        $indexer = new $className;
        if (!($indexer instanceof IndexerInterface)) {
            throw new InvalidObjectException(
                'Indexers must implement ' . IndexerInterface::class . ' - ' . $className . ' does not.'
            );
        }
        if (array_key_exists('configuration', $configuration)) {
            $indexer->setConfiguration($configuration['configuration']);
        }
        return $indexer;
    }
}
