<?php declare(strict_types=1);
namespace Newland\Toubiz\Search\Neos\Domain\Search\Mysql;

/*
 * This file is part of the "toubiz-search-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Doctrine\ORM\QueryBuilder;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Persistence\Doctrine\Repository;
use Newland\Toubiz\Search\Neos\Domain\Model\SearchQuery;
use Newland\Toubiz\Search\Neos\Domain\Search\AbstractSearch;
use Newland\Toubiz\Search\Neos\Domain\Search\SearchInterface;

/**
 * Search implementation for fulltext queries.
 *
 * This depends on a fulltext index on the "content" column to
 * be performant.
 *
 * @Flow\Scope("singleton")
 */
class FulltextSearch extends AbstractSearch implements SearchInterface
{
    /**
     * Search given query.
     *
     * @param SearchQuery $search
     * @param Repository $repository
     * @return QueryBuilder
     */
    public function getQuery(SearchQuery $search, Repository $repository): QueryBuilder
    {
        return $repository->createQueryBuilder('i')
                            ->where('MATCH (i.content) AGAINST (:full_search boolean) > 1')
                            ->setParameter('full_search', $this->buildSearchString($search->getQueryString()));
    }

    /**
     * Build search string for boolean search mode.
     *
     * @param string $queryString
     * @return string
     */
    protected function buildSearchString(string $queryString): string
    {
        $queryString = $this->stripInvalidCharacters($queryString);

        $searchWords = [];
        $words = explode(' ', $queryString);
        foreach ($words as $word) {
            $word = trim($word);

            if (empty($word)) {
                // Skip empty strings as just querying '*' is invalid.
                continue;
            }

            $group = [$word];

            if ($this->configuration['wordMatching']['before']) {
                $group[] = '*' . $word;
            }

            if ($this->configuration['wordMatching']['after']) {
                $group[] = $word . '*';
            }

            $searchWords[] = implode(' ', $group);
        }

        return implode(' ', $searchWords);
    }

    /**
     * Strips boolean search operators and other invalid
     * characters from the given string.
     *
     * @param string $string
     * @return string
     */
    protected function stripInvalidCharacters(string $string): string
    {
        $string = parent::stripInvalidCharacters($string);
        return str_replace(
            ['(', ')', '"', '+', '-', '~', '<', '>', '*'],
            '',
            $string
        );
    }
}
