<?php declare(strict_types=1);
namespace Newland\Toubiz\Search\Neos\ViewHelpers\Format;

/*
 * This file is part of the "toubiz-search-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\FluidAdaptor\Core\ViewHelper\AbstractViewHelper;
use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Search\Neos\Domain\Model\SearchQuery;
use function Safe\preg_replace;

/**
 * Text formatting view helper.
 *
 * Formats texts (raw, html) and highlights search words.
 *
 * @Flow\Scope("singleton")
 */
class TextViewHelper extends AbstractViewHelper
{
    /**
     * @param string $text
     * @param SearchQuery $searchQuery
     * @param integer|null $maxCharacters
     * @param string|null $append
     * @return string
     */
    public function render(
        ?string $text,
        SearchQuery $searchQuery,
        ?int $maxCharacters = null,
        ?string $append = null
    ): string {
        if (!$text) {
            return '';
        }
        $text = $this->sanitize($text);
        if ($maxCharacters) {
            $text = $this->shorten($text, $maxCharacters, $append);
        }
        return $this->highlight($text, $searchQuery->getSearchWords());
    }

    /**
     * Sanitizes the given string from unwanted contents.
     *
     * @param string $string
     * @return string
     */
    protected function sanitize(string $string): string
    {
        // Stripping HTML tags as HTML content may lead to broken views.
        $string = strip_tags($string);

        // Convert HTML entities to their corresponding characters.
        $string = html_entity_decode($string);

        return $string;
    }

    /**
     * Shortens the given string to $maxCharacters.
     *
     * @param string $string
     * @param int $maxCharacters
     * @param string|null $append
     * @return string
     */
    protected function shorten(string $string, int $maxCharacters, ?string $append = null): string
    {
        if (strlen($string) > $maxCharacters) {
            $chars = $append ? $maxCharacters - strlen($append) : $maxCharacters;
            $string = substr($string, 0, $chars);
            $string .= $append;
        }

        return $string;
    }

    /**
     * Highlight words in string.
     *
     * This is using preg_replace instead of str_replace or
     * str_ireplace, because both do not keep the original
     * casing of the word.
     *
     * @param string $string
     * @param array $words
     * @return string
     */
    protected function highlight(string $string, array $words): string
    {
        foreach ($words as $word) {
            $string = preg_replace(
                '/' . preg_quote($word, '/') . '/i',
                '<strong>$0</strong>',
                $string
            );
            if (!is_string($string)) {
                return '';
            }
        }

        return $string;
    }
}
