<?php declare(strict_types=1);
namespace Newland\Toubiz\Search\Neos\Indexer;

/*
 * This file is part of the "toubiz-search-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Neos\Flow\Persistence\PersistenceManagerInterface;
use Newland\Toubiz\Search\Neos\Domain\Model\SearchIndex;
use Newland\Toubiz\Search\Neos\Domain\Repository\SearchIndexRepository;
use function Safe\preg_replace;

/**
 * @Flow\Scope("singleton")
 */
abstract class AbstractIndexer
{
    /**
     * @var array
     */
    protected $configuration = [];

    /**
     * @var array
     */
    protected $index = [];

    /**
     * @var SearchIndexRepository
     * @Flow\Inject
     */
    protected $searchIndexRepository;

    /**
     * @var PersistenceManagerInterface
     * @Flow\Inject()
     */
    protected $persistenceManager;

    /** @var int */
    protected $indexed = 0;

    /** @var int */
    protected $persistEvery = 100;

    /**
     * @var callable[]
     */
    private $onProgress = [];

    public function onProgress(callable $block): void
    {
        $this->onProgress[] = $block;
    }

    protected function informProgress(int $numberProcessed, ?int $numberTotal): void
    {
        foreach ($this->onProgress as $listener) {
            $listener($numberProcessed, $numberTotal);
        }
    }

    public function name(): string
    {
        $parts = explode('\\', get_class($this));
        $name = (string) array_pop($parts);
        return str_replace('Indexer', '', $name);
    }

    public function setConfiguration(array $configuration = []): void
    {
        $this->configuration = $configuration;
    }

    protected function addToIndex(IndexRecordModification $modification): void
    {
        $this->index[] = $modification->getIdentifier();
        $modification->setSource($this);

        $this->searchIndexRepository->modifyItem(
            $modification->getIdentifier(),
            $modification->getScope(),
            function (SearchIndex $index) use ($modification) {
                $modification->mapInto($index);
            }
        );

        $this->persistenceManager->persistAll();
        if (++$this->indexed % $this->persistEvery === 0) {
            $this->persistenceManager->clearState();
        }
    }

    /**
     * Convert the given string to one that is suitable for searching
     * and/or representing.
     *
     * @param string $input
     * @return string
     */
    protected function convertToIndexableString(string $input): string
    {
        // Remove HTML tags.
        $string = strip_tags($input);

        // Replace html entities like `&nbsp;`.
        $string = html_entity_decode($string);

        // Replace newlines with a space to shorten the string.
        $string = preg_replace('/[\r\n]+/', ' ', $string);

        if (\is_array($string)) {
            return implode(' ', $string);
        }

        return $string;
    }
}
