<?php
namespace Newland\Toubiz\Search\Neos\Indexer;

/*
 * This file is part of the "toubiz-search-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\ContentRepository\Domain\Model\Node;
use Neos\Flow\Annotations as Flow;
use Neos\Eel\FlowQuery\FlowQuery;

/**
 * Document node indexer.
 *
 * Indexes document nodes by building an index entry for the document node
 * with all its child nodes.
 *
 * @Flow\Scope("singleton")
 */
class DocumentNodeIndexer extends AbstractNodeIndexer implements IndexerInterface
{
    /**
     * Entry point.
     *
     * Indexes document nodes.
     *
     * @return void
     */
    public function index()
    {
        $dimensions = $this->getContentDimensions();
        if (empty($dimensions)) {
            $this->indexContentDimension([]);
            return;
        }

        foreach ($dimensions as $dimensionName => $dimension) {
            foreach ($dimension['presets'] as $preset) {
                $this->indexContentDimension(
                    [
                        $dimensionName => $preset['values']
                    ]
                );
            }
        }
    }

    /**
     * Runs post-index tasks such as cleaning up old records.
     *
     * @return void
     */
    public function postIndex()
    {
        $this->searchIndexRepository->deleteObsolete($this->index, get_class($this));
    }

    /**
     * Fetches all nodes from a given dimension and indexes them.
     *
     * @param array $dimensions
     * @return void
     */
    protected function indexContentDimension($dimensions = [])
    {
        $context = $this->getContext(['dimensions' => $dimensions]);
        $documentNodes = (new FlowQuery([$context->getCurrentSiteNode()]))
            ->find('[instanceof Neos.Neos:Document]');

        foreach ($documentNodes as $node) {
            $indexedData = $this->indexNode($node);
            if ($indexedData) {
                $this->addToIndex(
                    [
                        'identifier' => $node->getIdentifier(),
                        'title' => $this->getTitleForNode($node),
                        'description' => $this->getDescriptionForNode($node),
                        'content' => $indexedData,
                        'scope' => $this->getScopeForContext($context),
                        'language' => $this->getLanguageForNode($node)
                    ]
                );
            }
        }
    }

    /**
     * Indexes given node.
     *
     * @param Node $node
     * @return string|boolean Indexed data, false if not indexed
     */
    protected function indexNode(Node $node)
    {
        if (!$this->nodeShouldBeIndexed($node)) {
            return false;
        }

        $indexData = [];
        $this->buildIndexDataForNode($node, $indexData);

        $childNodes = (new FlowQuery([$node]))->children('[instanceof Neos.Neos:Content]');
        foreach ($childNodes as $childNode) {
            $this->buildIndexDataForNode($childNode, $indexData);
        }

        return implode(' ', $indexData);
    }

    /**
     * Build index data for given node.
     *
     * @param Node $node
     * @param array $indexData
     * @return void
     */
    protected function buildIndexDataForNode(Node $node, &$indexData)
    {
        if (!$this->nodeShouldBeIndexed($node)) {
            return;
        }

        foreach ($node->getProperties() as $name => $value) {
            if (!$this->propertyShouldBeIndexed($node, $name, $value)) {
                continue;
            }

            $value = $this->stripIgnoredStrings($node, $value);
            $indexData[] = $this->convertToIndexableString($value);
        }
    }

    /**
     * Checks if a property should be indexed.
     *
     * @param Node $node The node object
     * @param string $name Property name
     * @param mixed $value Property value
     * @return bool
     */
    protected function propertyShouldBeIndexed(Node $node, $name, $value)
    {
        if (is_bool($value)) {
            // Boolean values do not provide any information.
            return false;
        }

        if (is_numeric($value)) {
            // Simple numeric values do not provide useful information.
            return false;
        }

        if (is_object($value)) {
            // Skip objects (like images, references, etc).
            return false;
        }

        if (is_array($value)) {
            // Skip arrays as they contain mostly internal data.
            return false;
        }

        if (preg_match('/^(node:\/\/)?[\w]{8}-[\w]{4}-[\w]{4}-[\w]{4}-[\w]{12}$/i', $value)) {
            // Skip persistence object identifiers;
            return false;
        }

        // Checking given node agains node configurations.
        foreach ($this->configuration['nodeTypes'] as $nodeType => $configuration) {
            if (!array_key_exists('ignoredProperties', $configuration)) {
                continue;
            }

            if ($node->getNodeType()->isOfType($nodeType)) {
                if (in_array($name, $configuration['ignoredProperties'])) {
                    // Ignored properties should not be indexed.
                    return false;
                }
            }
        }

        return true;
    }
}
