<?php

namespace Newland\Toubiz\Poi\Neos\Service;

use Neos\Cache\Frontend\FrontendInterface as CacheFrontend;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Cache\CacheManager;
use Neos\Flow\I18n\Translator;
use Newland\NeosCommon\Domain\Model\Topic;
use Newland\Toubiz\Poi\Neos\Domain\Repository\TopicRepository;
use Newland\Toubiz\Sync\Neos\Domain\Model\Attribute;
use Newland\Toubiz\Sync\Neos\Domain\Repository\AttributeRepository;

/**
 * @Flow\Scope("singleton")
 */
class AttributeTopicService
{
    const CACHE_NAME = 'Newland_Toubiz_Poi_Neos-Topics';
    const CACHE_KEY_TEMPLATE = 'TopicService__valid-attributes-:mainType';

    /**
     * @var Translator
     * @Flow\Inject()
     */
    protected $translator;

    /**
     * @var array
     * @Flow\InjectConfiguration(package="Newland.Toubiz.Poi.Neos", path="attributes.i18n")
     */
    protected $translationConfiguration;

    /**
     * @var CacheManager
     * @Flow\Inject()
     */
    protected $cacheManager;

    /**
     * @var AttributeRepository
     * @Flow\Inject()
     */
    protected $attributeRepository;

    /**
     * @var array
     * @Flow\InjectConfiguration(package="Newland.Toubiz.Poi.Neos", path="attributes")
     */
    protected $attributeConfiguration;

    /**
     * @param string $name
     * @param string|null $value
     * @return string
     */
    public function titleForAttribute(string $name, $value = null): string
    {
        $label = 'attribute.' . $name;
        if (\is_string($value)) {
            $label .= '.' . $value;
        }

        $configurations = $this->translationConfiguration;
        $configurations[] = [ 'package' => 'Newland.Toubiz.Poi.Neos' ];

        foreach ($configurations as $config) {
            $source = array_key_exists('source', $config) ? $config['source'] : 'Models/Topic';
            $package = $config['package'];
            $title = $this->translator->translateById($label, [], null, null, $source, $package);
            if ($title) {
                return $title;
            }
        }

        return $label;
    }

    public function getConfiguredIdentifiers($mainType = 'default'): array
    {
        return $this->cache()->get($this->attributeCacheKey($mainType)) ?:
            $this->generateAndCacheIdentifiers($mainType);
    }

    public function generateAndCacheIdentifiers($mainType = 'default'): array
    {
        $configured = $this->attributeConfiguration['articleTypes'][$mainType] ?? [];
        $prefix = TopicRepository::TYPE_ATTRIBUTE . ':';

        $attributes = [];
        $addToAttributesIfConfigured = function ($identifier) use (&$attributes, $configured) {
            if (\in_array($identifier, $configured, true)) {
                $attributes[$identifier] = true;
            }
        };

        foreach ($this->attributeRepository->findAll() as $attribute) {
            /** @var Attribute $attribute */

            $addToAttributesIfConfigured($prefix . $attribute->getName());

            $type = $attribute->getType();
            if ($type === Attribute::TYPE_STRING || $type === Attribute::TYPE_NUMBER) {
                $addToAttributesIfConfigured($prefix . $attribute->getName() . ':' . $attribute->getData());
            }
        }

        $identifiers = array_keys($attributes);
        $this->cache()->set($this->attributeCacheKey($mainType), $identifiers);

        return $identifiers;
    }

    private function attributeCacheKey(string $mainType): string
    {
        return str_replace(static::CACHE_KEY_TEMPLATE, ':mainType', $mainType);
    }

    private function cache(): CacheFrontend
    {
        return $this->cacheManager->getCache(static::CACHE_NAME);
    }
}
