<?php
namespace Newland\Toubiz\Poi\Neos\ViewHelpers\Widget\Controller;

/*
 * This file is part of the "toubiz-poi-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Neos\Flow\Log\Utility\LogEnvironment;
use Neos\FluidAdaptor\Core\Widget\AbstractWidgetController;
use Neos\Neos\Service\LinkingService;
use Newland\Toubiz\Poi\Neos\Domain\Repository\TopicRepository;
use Newland\Toubiz\Sync\Neos\Domain\Model\Article;
use Newland\Toubiz\Sync\Neos\Domain\Model\CityData;
use Newland\Toubiz\Sync\Neos\Exception\MissingDataException;
use Psr\Log\LoggerInterface;

/**
 * The widget controller for the city list widget.
 */
class LinkListController extends AbstractWidgetController
{

    /**
     * @var TopicRepository
     * @Flow\Inject()
     */
    protected $topicRepository;

    /**
     * @var array
     * @Flow\InjectConfiguration(package="Newland.Toubiz.Poi.Neos.cityDetails")
     */
    protected $cityDetailsConfiguration;

    /**
     * @var LinkingService
     * @Flow\Inject()
     */
    protected $linkingService;

    /**
     * @var LoggerInterface
     */
    protected $logger;

    public function indexAction(): void
    {
        $configuration = $this->cityDetailsConfiguration['links'];
        $links = [];
        $article = $this->widgetConfiguration['article'];
        if ($configuration['enabled'] === true) {
            $links = $configuration['data'];
            array_walk(
                $links,
                function (&$linkData) use ($article) {
                    $linkData = $this->prepareTargetUri($linkData, $article);
                    $linkData['translationPackage'] = $linkData['translationPackage'] ??
                        $this->cityDetailsConfiguration['defaultTranslation']['package'];
                    $linkData['translationSource'] = $linkData['translationSource'] ??
                        $this->cityDetailsConfiguration['defaultTranslation']['source'];
                }
            );
        }
        $this->view->assignMultiple(
            [
                'links' => $links,
                'article' => $article,
            ]
        );
    }

    private function prepareTargetUri(array $linkData, Article $article): ?array
    {
        if (!$linkData['target']) {
            return null;
        }
        $target = $linkData['target'];
        if (strpos($target, 'node://') === 0) {
            $pluginNamespace = $linkData['pluginNamespace'] ?? '--plugin';
            $linkData['target'] = $this->getUriFromNode($article, $linkData['target'], $pluginNamespace);
        } else {
            $linkData['target'] = $this->fillUriPlaceholders($article, $linkData['target']);
        }

        return $linkData;
    }

    /**
     * @param string $url
     * @param array $replacements
     * @return string
     * @todo Extract to helper package
     *
     */
    private function urlTemplate(string $url, array $replacements): string
    {
        return str_replace(
            array_keys($replacements),
            array_map('urlencode', array_values($replacements)),
            $url
        );
    }

    private function getUriFromNode(Article $article, string $nodeUri, string $pluginNamespace): string
    {
        $baseNode = $this->widgetConfiguration['node'];
        $node = $this->linkingService->convertUriToObject($nodeUri, $baseNode);
        $arguments = [
            $pluginNamespace => [
                'query' => [
                    'city' => $article->getPersistenceObjectIdentifier(),
                ],
            ],
        ];

        try {
            $uri = $this->linkingService->createNodeUri(
                $this->getControllerContext(),
                $node,
                $baseNode,
                null,
                false,
                $arguments
            );
        } catch (\Exception $exception) {
            $this->logger->error($exception->getMessage(), LogEnvironment::fromMethodName(__METHOD__));
            $uri = '';
        }

        return $uri;
    }

    private function fillUriPlaceholders(Article $article, string $targetUri): string
    {
        if ($article->getCityData() === null) {
            throw new MissingDataException('City article has no city data attached.', 1561279197);
        }

        $uri = $this->urlTemplate(
            $targetUri,
            $this->getUrlReplacements($article->getCityData())
        );
        return $uri;
    }

    private function getUrlReplacements(CityData $cityData): array
    {
        return [
            '{idToubiz}' => $cityData->getIdToubiz(),
            '{idTomas}' => $cityData->getIdTomas(),
        ];
    }
}
