<?php declare(strict_types=1);

namespace Newland\Toubiz\Poi\Neos\Tests\Unit\PageFrame;

use Neos\ContentRepository\Domain\Model\Node;
use Neos\ContentRepository\Domain\Model\NodeData;
use Neos\ContentRepository\Domain\Model\Workspace;
use Neos\Flow\Tests\FunctionalTestCase;
use Neos\Flow\Utility\Now;
use Neos\Neos\Domain\Service\ContentContext;
use Newland\NeosTestingHelpers\InteractsWithNodes;
use Newland\PageFrameProvider\Service\PageFrameContextService;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Poi\Neos\PageFrame\ArticleBreadcrumbProvider;
use Newland\Toubiz\Sync\Neos\Domain\Model\Article;
use Newland\Toubiz\Sync\Neos\Tests\Factory\ArticleFactory;

class ArticleBreadcrumbProviderTest extends FunctionalTestCase
{
    use InteractsWithNodes;
    protected static $testablePersistenceEnabled = true;

    /** @var Node[] */
    protected $nodes;

    /** @var ArticleBreadcrumbProvider */
    protected $subject;

    /** @var Article */
    protected $article;

    public function setUp(): void
    {
        parent::setUp();

        $this->subject = $this->objectManager->get(ArticleBreadcrumbProvider::class);
        $this->article = (new ArticleFactory($this->objectManager))
                ->create([ 'mainType' => ArticleConstants::TYPE_TOUR ]);
        $this->objectManager->get(PageFrameContextService::class)
            ->setPluginArguments([ 'article' => $this->article->getPersistenceObjectIdentifier() ]);

        $this->nodes = [
            'site-test' => $this->initializeNode('/sites/site-test'),
            'foo' => $this->initializeNode('/sites/site-test/foo'),
            'bar' => $this->initializeNode('/sites/site-test/bar'),
            'baz' => $this->initializeNode('/sites/site-test/baz'),
        ];
    }


    public function testFetchesNodesAsConfigured(): void
    {
        $this->setupBreadcrumbConfig([
            'site-test' => [ '/sites/site-test/foo', '/sites/site-test/bar' ],
        ]);

        $nodes = $this->subject->selectBreadCrumbs($this->nodes['site-test'], []);
        $this->assertEquals('site-test', $nodes[0]->getName());
        $this->assertEquals('foo', $nodes[1]->getName());
        $this->assertEquals('bar', $nodes[2]->getName());
    }

    public function testUsesFallbackIfNothingConfigured(): void
    {
        $this->setupBreadcrumbConfig([
            'site-another' => [ '/sites/site-test/foo', '/sites/site-test/bar' ],
            '_fallback' => [ '/sites/site-test/baz' ],
        ]);

        $nodes = $this->subject->selectBreadCrumbs($this->nodes['site-test'], []);
        $this->assertEquals('site-test', $nodes[0]->getName());
        $this->assertEquals('baz', $nodes[1]->getName());
    }

    public function testUsesOldStyleConfigurationIfFallbackNotAvailable(): void
    {
        $this->setupBreadcrumbConfig([
             'site-another' => [ '/sites/site-test/bar' ],
             '/sites/site-test/baz',
             '/sites/site-test/foo'
         ]);

        $nodes = $this->subject->selectBreadCrumbs($this->nodes['site-test'], []);
        $this->assertEquals('site-test', $nodes[0]->getName());
        $this->assertEquals('baz', $nodes[1]->getName());
        $this->assertEquals('foo', $nodes[2]->getName());
    }

    private function setupBreadcrumbConfig(array $breadcrumb): void
    {
        $this->inject(
            $this->subject,
            'typeConfiguration',
            [
                ArticleConstants::TYPE_TOUR => [
                    'uriSegment' => 'foo/bar',
                    'breadcrumb' => $breadcrumb,
                ],
            ]
        );
    }

}