<?php declare(strict_types=1);

namespace Newland\Toubiz\Poi\Neos\ViewHelpers;

use Neos\Flow\Mvc\ActionRequest;
use Neos\FluidAdaptor\Core\ViewHelper\AbstractTagBasedViewHelper;
use Neos\Flow\Annotations as Flow;
use Newland\NeosIcons\IconUrlService;
use Newland\Toubiz\Poi\Neos\Service\MapCompatibility\ApiKeyService;

class StaticMapViewHelper extends AbstractTagBasedViewHelper
{

    /**
     * @var array
     * @Flow\InjectConfiguration(package="Newland.Toubiz.Poi.Neos", path="staticMap")
     */
    protected $staticMapConfiguration;

    /**
     * @var ApiKeyService
     * @Flow\Inject()
     */
    protected $apiKeyService;

    /**
     * @var IconUrlService
     * @Flow\Inject()
     */
    protected $iconUrlService;

    public function initializeArguments(): void
    {
        $this->registerArgument('latitude', 'float', 'Latitude of marker on map', true);
        $this->registerArgument('longitude', 'float', 'Longitude of marker on map', true);
        $this->registerArgument('width', 'integer', 'Width of map in pixels', false, 800);
        $this->registerArgument('height', 'integer', 'Height of map in pixels', false, 600);
        $this->registerArgument('zoom', 'integer', 'Zoom level of map (0-100)', false, 75);
        $this->registerArgument('mapType', 'string', 'Type of map to render', false, null);
    }

    public function render(): string
    {
        $mapType = $this->arguments['mapType'] ?? $this->staticMapConfiguration['mapType'] ?? null;

        if ($mapType === null) {
            return '<!-- Static map not rendered: No map type defined. -->';
        }

        /** @var ActionRequest $request */
        $request = $this->controllerContext->getRequest();
        $iconUrl = $this->iconUrlService->buildMarkerIconUrl(
            $this->staticMapConfiguration['style'] ?? [],
            $request
        );
        $this->addApiKeysToTags();
        $this->tag->addAttributes(
            [
                'class' => 'cursor-pointer',
                'data-toubiz-static-map' => '',
                'data-toubiz-static-map.map-type' => $mapType,
                'data-toubiz-static-map.map.zoom' => $this->arguments['zoom'],
                'data-toubiz-static-map.width' => $this->arguments['width'],
                'data-toubiz-static-map.height' => $this->arguments['height'],
                'data-toubiz-static-map.markers.0.icon-url' => $iconUrl,
                'data-toubiz-static-map.markers.0.latitude' => $this->arguments['latitude'],
                'data-toubiz-static-map.markers.0.longitude' => $this->arguments['longitude'],
            ]
        );

        $this->tag->forceClosingTag(true);
        return $this->tag->render();
    }

    private function addApiKeysToTags(): void
    {
        $apiKeys = $this->apiKeyService->getApiKeys($this->controllerContext->getRequest());

        if (array_key_exists('googleMaps', $apiKeys)) {
            $this->tag->addAttribute(
                'data-toubiz-static-map.map.google-maps-api-key',
                $apiKeys['googleMaps']
            );
        }

        if (array_key_exists('toursprung', $apiKeys)) {
            $this->tag->addAttribute(
                'data-toubiz-static-map.map.toursprung-api-key',
                $apiKeys['toursprung']
            );
        }
    }
}
