<?php
namespace Newland\Toubiz\Poi\Neos\Search\Indexer;

/*
 * This file is part of the "toubiz-poi-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Newland\Contracts\Neos\Search\IndexerInterface;
use Newland\Contracts\Neos\Search\IndexingBackend;
use Newland\Contracts\Neos\Search\IndexRecordModification;
use Newland\Contracts\Neos\Search\ProgressHandler;
use Newland\Toubiz\Sync\Neos\Domain\Model\Article;
use Newland\Toubiz\Sync\Neos\Domain\Repository\ArticleRepository;

/**
 * Article indexer.
 *
 * Indexes articles for the search.
 *
 * @Flow\Scope("singleton")
 */
class ArticleIndexer implements IndexerInterface
{
    public const SCOPE = 'article';

    /**
     * @var ArticleRepository
     * @Flow\Inject()
     */
    protected $articleRepository;

    /** @var int|null */
    private $total;

    /** @var string[] */
    private $indexed = [];

    /** @var array */
    protected $configuration;

    public function index(IndexingBackend $backend, ProgressHandler $progressHandler): void
    {
        $this->articleRepository->withoutLanguageHandling(function () use ($backend, $progressHandler) {
            $this->total = $this->articleRepository->countAll();

            foreach ($this->articleRepository->findAllIterator() as $articles) {
                foreach ($articles as $article) {
                    $modification = $this->articleToModification($article);
                    $backend->createOrUpdateIndexEntry($modification);
                    $this->indexed[] = $modification->getIdentifier();
                    $progressHandler->informProgress('Articles', \count($this->indexed), $this->total);
                }
            }
        });
    }

    public function postIndex(IndexingBackend $backend): void
    {
        $backend->deleteObsoleteIndexEntries($this->indexed);
    }

    public function setConfiguration(array $configuration = []): void
    {
        $this->configuration = $configuration;
    }

    private function articleToModification(Article $article): IndexRecordModification
    {
        $scope = static::SCOPE;
        $mainType = $article->getMainType();

        switch ($mainType) {
            case 0:
                $scope = 'attraction';
                break;
            case 1:
                $scope = 'gastronomy';
                break;
            case 2:
                $scope = 'tour';
                break;
            case 3:
                $scope = 'lodging';
                break;
            case 4:
                $scope = 'directMarketer';
                break;
            case 5:
                $scope = 'congressLocation';
                break;
            case 6:
                $scope = 'city';
                break;
        }

        $content = [
            $article->getName(),
            $article->getAbstract(),
            $article->getDescription(),
        ];

        return new IndexRecordModification(
            (string) $article->getPersistenceObjectIdentifier(),
            $article->getName(),
            (string) $article->getAbstract(),
            trim(implode(' ', $content)),
            $scope,
            $article->getLanguage()
        );
    }
}
