<?php
namespace Newland\Toubiz\Poi\Neos\Service;

/*
 * This file is part of the "toubiz-poi-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Neos\ContentRepository\Domain\Model\Node;
use Neos\Flow\I18n\Locale;
use Neos\Flow\I18n\Translator;
use Neos\Utility\Arrays;
use Newland\Toubiz\Poi\Neos\Domain\Model\Tag;
use Newland\NeosCommon\Domain\Model\Topic;
use Newland\Toubiz\Poi\Neos\Domain\Repository\TopicRepository;

/**
 * Class TagTranslationService
 *
 * @package Newland\Toubiz\Poi\Neos\Service
 * @Flow\Scope("singleton")
 */
class TagTranslationService
{
    /**
     * @var Node
     */
    protected $node;

    /**
     * @var Translator
     * @Flow\Inject
     */
    protected $translator;

    /**
     * @var array
     * @Flow\InjectConfiguration(package="Newland.Toubiz.Poi.Neos", path="filter.options")
     */
    protected $options;

    /**
     * @var string
     */
    private $locale;

    public function getTitle(Tag $tag, Node $node)
    {
        $this->parseLocaleFromNode($node);

        if (!empty($tag->getTitle())) {
            return $tag->getTitle();
        } elseif ($tag->getAttributeType() === Tag::TYPE_RANGE) {
            return $this->getRangeTitle($tag);
        } elseif ($tag->getAttributeType() === Tag::TYPE_STRING) {
            return $this->getValueTitle($tag);
        } else {
            return $this->getAttributeTitle($tag->getAttribute());
        }
    }

    /**
     * For manual translation we need to find the current language.
     */
    private function parseLocaleFromNode(Node $node)
    {
        // This only needs to be parsed once because the locale should not change during a single request.
        if (!$this->locale) {
            $dimensions = $node->getDimensions();
            if (isset($dimensions['language'])
                && is_array($dimensions['language'])
                && !empty($dimensions['language'])
            ) {
                $this->locale = new Locale($dimensions['language'][0]);
            }
        }
    }

    /**
     * @param string $attribute
     * @return string
     */
    private function getAttributeTitle(string $attribute): string
    {
        return $this->translate(TopicRepository::TYPE_ATTRIBUTE . '.' . str_replace(':', '.', $attribute));
    }

    /**
     * @param string $id
     * @return string
     */
    private function translate(string $id): string
    {
        return (string) $this->translator->translateById(
            $id,
            [],
            null,
            $this->locale,
            'Models/Topic',
            'Newland.Toubiz.Poi.Neos'
        );
    }

    private function getUnit(string $attribute): string
    {
        return Arrays::getValueByPath($this->options, $attribute . '.unit') ?? '';
    }

    private function applyFactor(string $attribute, float $value): string
    {
        $factor = Arrays::getValueByPath($this->options, $attribute . '.factor') ?? 1;
        $steps = Arrays::getValueByPath($this->options, $attribute . '.steps') ?? 1;
        $numberOfDecimals = 1 / $steps;
        $precision = log($numberOfDecimals, 10);
        return number_format($value * $factor, $precision, '.', '');
    }

    private function getRangeTitle(Tag $tag): string
    {
        $title = $this->getAttributeTitle($tag->getAttribute());

        if (preg_match('/(?<min>\d+)(\.\d+)?;(?<max>\d+)(\.\d+)?/', $tag->getValue(), $matches)) {
            $title = sprintf(
                '%s %s - %s%s',
                $this->getAttributeTitle($tag->getAttribute()),
                $this->applyFactor($tag->getAttribute(), (float) $matches['min']),
                $this->applyFactor($tag->getAttribute(), (float) $matches['max']),
                $this->getUnit($tag->getAttribute())
            );
        }

        return $title;
    }

    private function getValueTitle(Tag $tag): string
    {
        return sprintf(
            '%s %s',
            $this->getAttributeTitle($tag->getAttribute()),
            $this->getAttributeTitle($tag->getAttribute() . '.' . $tag->getValue())
        );
    }
}
