<?php

/*
 * This file is part of the "toubiz-poi-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

namespace Newland\Toubiz\Poi\Neos\ViewHelpers\Filter\Controller;

use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\I18n\Translator;
use Neos\Flow\ObjectManagement\ObjectManagerInterface;
use Neos\FluidAdaptor\Core\Widget\AbstractWidgetController;
use Newland\Toubiz\Poi\Neos\Domain\Model\Tag;
use Newland\Toubiz\Poi\Neos\Domain\Model\Topic;
use Newland\Toubiz\Sync\Neos\Domain\Filter\CategoryFilter;
use Newland\Toubiz\Sync\Neos\Domain\Model\Category;
use Newland\Toubiz\Sync\Neos\Domain\Repository\CategoryRepository;

class TagController extends AbstractWidgetController
{
    /**
     * @var CategoryRepository
     * @Flow\Inject
     */
    protected $categoryRepository;

    /**
     * @var CategoryFilter
     * @Flow\Inject
     */
    protected $categoryFilter;

    /**
     * @var ObjectManagerInterface
     * @Flow\Inject
     */
    protected $objectManager;

    /**
     * @var Translator
     * @Flow\Inject
     */
    protected $translator;
    /**
     * @var NodeInterface
     */
    protected $node;
    /**
     * @var array
     */
    private $tags;
    /**
     * @var array
     */
    private $parameters = [];

    public function indexAction()
    {
        $this->node = $this->widgetConfiguration['node'];
        $this->parameters = $this->widgetConfiguration['queryParameters'];

        $this->parseCategories();
        $this->parseAttributes();

        $this->view->assignMultiple(
            [
                'tags' => $this->tags,
            ]
        );

    }

    private function parseCategories()
    {
        foreach ($this->getCategories() as $category) {
            $tag = $this->objectManager->get(Tag::class);
            $tag->setTitle($category->getTitle());
            $topicId = Topic::TYPE_CATEGORY . ':' . $category->getPersistenceObjectIdentifier();
            $tag->setIdentifier($this->node->getIdentifier() . '_' . $topicId);
            $this->tags[] = $tag;
        }
    }

    /**
     * @return Category[]
     */
    private function getCategories(): array
    {
        $categories = [];
        if (array_key_exists('categories', $this->parameters)) {
            foreach ((array) $this->parameters['categories'] as $identifier) {
                $categories[] = $this->categoryRepository->findByIdentifier($identifier);
            }
        }

        return (array) $categories;
    }

    private function parseAttributes()
    {
        foreach ($this->getAttributes() as $attribute) {
            $tag = $this->objectManager->get(Tag::class);
            $tag->setTitle($this->getAttributeTitle($attribute));
            $topicId = Topic::TYPE_ATTRIBUTE . ':' . $attribute;
            $tag->setIdentifier($this->node->getIdentifier() . '_' . $topicId);
            $this->tags[] = $tag;
        }
    }

    /**
     * @return string[]
     */
    private function getAttributes(): array
    {
        $attributes = [];
        if (array_key_exists('attributes', $this->parameters)) {
            foreach ($this->parameters['attributes'] as $key => $attribute) {
                if ($key === 'properties') {
                    $attributes = array_merge($attributes, $this->getPropertyAttributes($attribute));
                } elseif ($key === 'labels') {
                    $attributes = array_merge($attributes, $this->getLabelAttributes($attribute));
                } elseif ($this->isRangeAttribute($attribute)) {
                    $attributes[] = $key;
                }
            }
        }

        return $attributes;
    }

    private function getPropertyAttributes($attribute): array
    {
        return array_map(
            function (string $property) {
                return 'properties:' . $property;
            },
            array_values($attribute['in'])
        );
    }

    private function getLabelAttributes(array $attribute)
    {
        return array_map(
            function (string $label) {
                return 'labels:' . $label;
            },
            array_values($attribute['in'])
        );
    }

    private function isRangeAttribute($attribute): bool
    {
        return is_array($attribute)
            && array_key_exists('min', $attribute)
            && array_key_exists('max', $attribute)
            && (!empty($attribute['min']) || !empty($attribute['max']));
    }

    /**
     * @param string $attribute
     * @return string
     */
    private function getAttributeTitle(string $attribute): string
    {
        return $this->translate(Topic::TYPE_ATTRIBUTE . '.' . str_replace(':', '.', $attribute));
    }

    /**
     * @param string $id
     * @return string
     */
    private function translate(string $id): string
    {
        return (string) $this->translator->translateById(
            $id,
            [],
            null,
            null,
            'Models/Topic',
            'Newland.Toubiz.Poi.Neos'
        );
    }
}
