<?php

namespace Newland\Toubiz\Poi\Neos\Domain\Repository;

use Neos\Flow\ObjectManagement\ObjectManager;
use Neos\Utility\Arrays;
use Newland\Toubiz\Poi\Neos\Domain\Model\Topic;
use Newland\Toubiz\Poi\Neos\Service\TopicService;
use Newland\Toubiz\Sync\Neos\Domain\Filter\ArticleFilter;
use Newland\Toubiz\Sync\Neos\Domain\Filter\CategoryFilter;
use Newland\Toubiz\Sync\Neos\Domain\Model\Category;
use Newland\Toubiz\Sync\Neos\Domain\Repository\ArticleRepository;
use Newland\Toubiz\Sync\Neos\Domain\Repository\CategoryRepository;
use Neos\Flow\Annotations as Flow;

/**
 * @Flow\Scope("singleton")
 */
class TopicRepository
{

    /**
     * @var CategoryRepository
     * @Flow\Inject()
     */
    protected $categoryRepository;

    /**
     * @var ArticleRepository
     * @Flow\Inject()
     */
    protected $articleRepository;

    /**
     * @var ObjectManager
     * @Flow\Inject
     */
    protected $objectManager;

    /**
     * @var TopicService
     * @Flow\Inject()
     */
    protected $topicService;

    /**
     * @var array
     * @Flow\InjectConfiguration(path="attributes")
     */
    protected $attributeConfiguration;

    /**
     * @param string|int|null $articleType
     * @return Topic[]
     */
    public function findAll($articleType = null): iterable
    {
        return array_merge(
            $this->findAllCategories($articleType),
            $this->findAllAttributes($articleType)
        );
    }

    /**
     * @param string|int|null $articleType
     * @return Topic[]
     */
    public function findAllCategories($articleType = null): iterable
    {
        $filter = new CategoryFilter();
        $filter->setOrderBy([ 'category.title' => 'ASC' ]);

        if ($articleType !== null) {
            $filter->setArticleMainType($articleType);
        }

        return array_map(
            function (Category $category) {
                return $this->topicFromCategory($category);
            },
            $this->categoryRepository->findByFilter($filter)
        );
    }

    /**
     * @param string|int|null $articleType
     * @return Topic[]
     */
    public function findAllAttributes($articleType = null): iterable
    {
        $configPath = 'articleTypes.' . ($articleType ?? 'default');
        $attributeConfiguration = Arrays::getValueByPath($this->attributeConfiguration, $configPath) ?? [];

        $attributes = [];
        foreach ($attributeConfiguration as $key => $configuration) {
            switch ($configuration['type']) {

                case 'checkboxes':
                    foreach ($configuration['options'] as $option) {
                        $attributes[] = $this->findByAttribute($key . ':' . $option);
                    }
                    break;

                case 'range':
                default:
                    $attributes[] = $this->findByAttribute($key);
            }
        }

        return $attributes;
    }

    /**
     * @param array $identifiers
     * @return Topic[]
     */
    public function findByIdentifiers(array $identifiers): iterable
    {
        $topics = [];

        foreach ($identifiers as $identifier) {
            $topic = $this->findByIdentifier($identifier);
            if ($topic) {
                $topics[] = $topic;
            }
        }

        return $topics;
    }

    /**
     * @param string $identifier
     * @return Topic|null
     */
    public function findByIdentifier(string $identifier)
    {
        list($type, $id) = explode(':', $identifier, 2);

        switch ($type) {

            case Topic::TYPE_CATEGORY:
                return $this->findByCategory($id);

            case Topic::TYPE_ATTRIBUTE:
                return $this->findByAttribute($id);

        }

        return null;
    }

    /**
     * @param string $attributeString
     * @return Topic|null
     */
    public function findByAttribute(string $attributeString)
    {
        $exploded = explode(':', $attributeString);
        $attribute = $exploded[0];
        $value = \count($exploded) > 1 ? $exploded[1] : null;

        // TODO Move to article repository / filter.
        $query = $this->articleRepository->createQueryBuilder('article');
        $query->innerJoin('article.attributes', 'attributes')
            ->where(
                $query->expr()->andX(
                    $query->expr()->eq('attributes.name', ':name'),
                    $query->expr()->eq('attributes.data', ':value')
                )
            )->setParameters(
                [
                    'name' => $attribute,
                    'value' => $value === true ? 'true' : $value,
                ]
            );

        $title = $this->topicService->titleForAttribute($attribute, $value) ?: $attributeString;
        return (new Topic())
            ->setType(Topic::TYPE_ATTRIBUTE)
            ->setIdentifier(Topic::TYPE_ATTRIBUTE . ':' . $attributeString)
            ->setTitle($title)
            ->setArticleGetter(
                function () use ($query) {
                    return $query->setMaxResults(11)->getQuery()->execute();
                }
            )
            ->setTotalCount((clone $query)->select('COUNT(article) AS total')->getQuery()->execute()[0]['total'])
            ->setValue($value);
    }


    /**
     * @param array $categoryIds
     * @return Topic[]
     */
    public function findByCategories(array $categoryIds): iterable
    {
        return array_map(
            function (string $id) {
                return $this->findByCategory($id);
            },
            $categoryIds
        );
    }

    /**
     * @param string $categoryId
     * @return Topic|null
     */
    public function findByCategory(string $categoryId)
    {
        $category = $this->categoryRepository->findByIdentifier($categoryId);
        if (!$category) {
            return null;
        }

        return $this->topicFromCategory($category);
    }

    public function topicFromCategory(Category $category): Topic
    {
        $filter = new ArticleFilter();
        $filter->setCategories([ $category->getPersistenceObjectIdentifier() ]);
        $filter->setLimit(11);

        return (new Topic())
            ->setType(Topic::TYPE_CATEGORY)
            ->setIdentifier(Topic::TYPE_CATEGORY . ':' . $category->getPersistenceObjectIdentifier())
            ->setTitle($category->getTitle())
            ->setValue($category->getPersistenceObjectIdentifier())
            ->setTotalCount($this->articleRepository->countByFilter($filter, 11)['items'])
            ->setConfiguration([ 'type' => 'category' ])
            ->setArticleGetter(
                function () use ($filter) {
                    return $this->articleRepository->findByFilter($filter);
                }
            );
    }
}
