<?php
namespace Newland\Toubiz\Poi\Neos\Controller;

/*
 * This file is part of the "toubiz-poi-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\ContentRepository\Domain\Model\Node;
use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Sync\Neos\Domain\Filter\ArticleFilter;
use Newland\Toubiz\Sync\Neos\Domain\Filter\CategoryFilter;
use Newland\Toubiz\Sync\Neos\Domain\Model\Article;
use Newland\Toubiz\Sync\Neos\Domain\Repository\ArticleRepository;
use Newland\Toubiz\Sync\Neos\Domain\Repository\CategoryRepository;

/**
 * Point of interests controller.
 *
 * @Flow\Scope("singleton")
 */
class PointOfInterestsController extends AbstractActionController
{

    /**
     * @Flow\Inject
     * @var ArticleRepository
     */
    protected $articleRepository;

    /**
     * @Flow\Inject
     * @var CategoryRepository
     */
    protected $categoryRepository;

    /**
     * @var array
     * @Flow\InjectConfiguration("filter.sections")
     */
    protected $filterSections;

    /**
     * Teaser action.
     *
     * Renders a single teaser.
     *
     * @return void
     */
    public function teaserAction()
    {
        if (array_key_exists('article', $this->properties)) {
            $this->view->assign('article', $this->articleRepository->find($this->properties['article']));
        }
        $this->view->assignMultiple(
            [
                'wrapStart' => $this->request->getInternalArgument('__wrapStart'),
                'wrapEnd' => $this->request->getInternalArgument('__wrapEnd'),
            ]
        );
    }

    /**
     * Teaser list action.
     *
     * Renders a list of teasers.
     *
     * @return void
     */
    public function teaserListAction()
    {
        $filter = new ArticleFilter;
        if (array_key_exists('categoryFilter', $this->properties)) {
            $filter->setCategories($this->properties['categoryFilter']);
        }
        if (array_key_exists('recordLimit', $this->properties)) {
            $filter->setLimit($this->properties['recordLimit']);
        }

        $filter->setCurrentSite($this->currentSite->getNodeName());

        $articles = $this->articleRepository->findByFilter($filter);
        $displayStyle = $this->properties['displayStyle'];
        $this->view->assign('articles', $articles);
        $this->view->assign('enableSwiper', $displayStyle === 'swiper');
    }

    /**
     * List action.
     *
     * Lists articles and allows filtering.
     *
     * @return void
     */
    public function listAction()
    {

        $articleFilter = new ArticleFilter;
        $articleFilter->setLimit($this->configuration['resultsPerPage']);

        if (array_key_exists('articleType', $this->properties)) {
            $articleFilter->setMainType((int) $this->properties['articleType']);
        }
        $articleFilter->setCurrentSite($this->currentSite->getNodeName());

        $this->view->assignMultiple(
            [
                'articles' => $this->articleRepository->findByFilter($articleFilter),
                'pagination' => [
                    'isFirst' => true,
                    'page' => 1,
                    'isLast' => false,
                    'count' => $this->articleRepository->countByFilter($articleFilter, $this->configuration['resultsPerPage']),
                ],
            ]
        );

        $categoryFilter = new CategoryFilter;
        if (array_key_exists('articleType', $this->properties)) {
            $categoryFilter->setArticleMainType($this->properties['articleType']);
        }
        $categoryFilter->setOrderBy([ 'category.title' => 'asc' ]);
        $categories = $this->categoryRepository->findByFilter($categoryFilter);
        $this->view->assign('categories', $categories);
        $this->createStandaloneView('PointOfInterests/FilterList');

    }


    /**
     * Filter list action.
     *
     * Returns articles based on given filters.
     *
     * @param Node $node
     * @param array $query The filter query.
     * @return void
     */
    public function filterListAction(Node $node, array $query = [])
    {
        $this->initializeAction($node);
        $articleFilter = $this->initializeFilter($node, $query);

        if (array_key_exists('page', $query) && !empty($query['page'])) {
            $page = (int) $query['page'];
            $articleFilter->setOffset(($page - 1) * $this->configuration['resultsPerPage']);
        } else {
            $page = 1;
        }

        $articles = $this->articleRepository->findByFilter($articleFilter);
        $articleCounts = $this->articleRepository->countByFilter($articleFilter, $this->configuration['resultsPerPage']);

        $this->view->assignMultiple(
            [
                'articles' => $articles,
                'pagination' => [
                    'isFirst' => ($page == 1),
                    'page' => $page,
                    'isLast' => $articleCounts['pages'] <= $page,
                    'count' => $articleCounts,
                ],
                'query' => $query,
                'node' => $node,
            ]
        );
    }

    /**
     * Show action.
     *
     * Displays an article.
     *
     * @param Article $article
     * @return void
     */
    public function showAction(Article $article = null)
    {
        $this->view->assign('url', (string) $this->request->getHttpRequest()->getUri());
        $this->view->assign('article', $article);
        if (
            $article &&
            $article->getMainAddress() &&
            $article->getMainAddress()->getHasGeocoordinates()
        ) {
            $onSiteFilter = new ArticleFilter;
            $onSiteFilter->setLatitude($article->getMainAddress()->getLatitude())
                ->setLongitude($article->getMainAddress()->getLongitude())
                ->setCoordinatePrecision(4)
                ->setExcludes([ $article ]);

            $this->view->assign(
                'onSiteArticles',
                $this->articleRepository->findByFilter($onSiteFilter)
            );
        }

        if ($article->getMainType() == ArticleConstants::TYPE_TOUR) {
            $this->assignSimilarTours($article);
        }
    }

    /**
     * Assigns similar tours to the given one to the view.
     *
     * @param Article $tour
     * @return void
     */
    protected function assignSimilarTours(Article $tour)
    {
        $filter = new ArticleFilter;
        $filter->setCategoryCollection($tour->getCategories())
            ->setExcludes([ $tour ])
            ->setLimit(4);

        $this->view->assign(
            'similarTours',
            $this->articleRepository->findByFilter($filter)
        );
    }
}
