<?php
namespace Newland\Toubiz\Poi\Neos\Controller;

/*
 * This file is part of the "toubiz-poi-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\ContentRepository\Domain\Model\Node;
use Neos\Eel\FlowQuery\FlowQuery;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\I18n\Locale;
use Neos\Flow\I18n\Service;
use Neos\FluidAdaptor\View\StandaloneView;
use Neos\Flow\Mvc\Controller\ActionController;
use Neos\Flow\Mvc\View\ViewInterface;
use Neos\Neos\Domain\Model\Site;
use Newland\Toubiz\Sync\Neos\Domain\Filter\ArticleFilter;

/**
 * Abstract action controller.
 *
 * @Flow\Scope("singleton")
 */
abstract class AbstractActionController extends ActionController
{

    /**
     * The configuration for this package.
     *
     * @Flow\InjectConfiguration
     * @var array
     */
    protected $configuration;

    /**
     * The configuration for the sync package.
     *
     * @Flow\InjectConfiguration(package="Newland.Toubiz.Sync.Neos")
     * @var array
     */
    protected $syncConfiguration;

    /**
     * @var Node The current node.
     */
    protected $node;

    /**
     * @var Site
     */
    protected $currentSite;

    /**
     * @var Node The target node for linking records.
     */
    protected $targetNode;

    /**
     * @var array
     */
    protected $properties;

    /**
     * @Flow\Inject
     * @var Service
     */
    protected $i18nService;

    /**
     * Action initializer.
     *
     * Called before the called action. This may be called
     * manually inside an action that is working outside of
     * the default node/context, like an AJAX action.
     *
     * @param Node|null $node
     * @return void
     */
    protected function initializeAction(Node $node = null)
    {
        if ($node) {
            $this->node = $node;
        } else {
            $this->node = $this->request->getInternalArgument('__node');
        }

        if (!$this->node) {
            return;
        }

        $this->properties = $this->node->getProperties();
        $this->currentSite = $this->node->getContext()->getCurrentSite();
        $siteNode = $this->node->getContext()->getCurrentSiteNode();

        if (array_key_exists('articleType', $this->properties)) {
            $targetNodeQuery = new FlowQuery([ $siteNode ]);
            $this->targetNode = $targetNodeQuery->find(
                '[instanceof Newland.Toubiz.Poi.Neos:List][articleType *= "' . $this->properties['articleType'] . '"]'
            )->parents('[instanceof Neos.NodeTypes:Page]')->get(0);
        }
    }


    protected function initializeFilter(Node $node, array $query): ArticleFilter
    {

        $articleFilter = new ArticleFilter;
        $articleFilter->setLimit($this->configuration['resultsPerPage']);

        if (array_key_exists('articleType', $this->properties)) {
            $articleFilter->setMainType($this->properties['articleType']);
        }
        if (array_key_exists('categories', $query) && is_array($query['categories'])) {
            $articleFilter->setCategories($query['categories']);
        }


        if (array_key_exists('attributes', $query)) {
            foreach ($query['attributes'] as $attribute => $type) {
                if (array_key_exists('in', $type)) {
                    $articleFilter->setAttributesIn(
                        [
                            $attribute => [
                                'in' => $type['in'],
                            ],
                        ]
                    );
                } else {
                    if ($this->isValidRange($type)) {
                        $articleFilter->setAttributesRange(
                            [
                                $attribute => [
                                    'max' => $type['max'],
                                    'min' => $type['min'],
                                ],
                            ]
                        );
                    }
                }
            }
        }
        return $articleFilter;
    }

    /**
     * View initializer.
     *
     * Called before a view is rendered.
     *
     * @param ViewInterface $view
     * @return void
     */
    protected function initializeView(ViewInterface $view)
    {
        $view->assignMultiple(
            [
                'configuration' => $this->configuration,
                'node' => $this->node,
                'targetNode' => $this->targetNode,
            ]
        );

        // OA tracking requires api credentials.
        if (array_key_exists('Outdooractive/Api', $this->syncConfiguration['services'])) {
            $this->view->assign(
                'outdooractiveCredentials',
                $this->syncConfiguration['services']['Outdooractive/Api']
            );
        }
    }

    /**
     * Creates a fluid standalone view.
     *
     * @param string $template Path to the template to render.
     * @return StandaloneView
     */
    protected function createStandaloneView($template)
    {
        $view = new StandaloneView;
        $this->initializeView($view);
        $view->setTemplateRootPaths([ 'resource://Newland.Toubiz.Poi.Neos/Private/Templates' ]);
        $view->setPartialRootPaths([ 'resource://Newland.Toubiz.Poi.Neos/Private/Partials' ]);
        $view->setTemplate($template);

        $dimensions = $this->node->getContext()->getDimensions();
        if (array_key_exists('language', $dimensions) && $dimensions['language'] !== []) {
            $currentLocale = new Locale($dimensions['language'][0]);
            $this->i18nService->getConfiguration()->setCurrentLocale($currentLocale);
            $this->i18nService->getConfiguration()->setFallbackRule(
                [
                    'strict' => false,
                    'order' => array_reverse($dimensions['language']),
                ]
            );
        }

        return $view;
    }

    private function isValidRange($type): bool
    {
        return is_array($type)
            && array_key_exists('max', $type)
            && array_key_exists('min', $type)
            && !empty($type['max'])
            && !empty($type['min']);
    }
}
