<?php
namespace Newland\Toubiz\Poi\Neos\Tests\Routing;

use Neos\Flow\Mvc\Routing\Dto\ResolveResult;
use Neos\Flow\Tests\FunctionalTestCase;
use Newland\PageFrameProvider\Service\PageFrameLinkingService;
use Newland\Toubiz\Poi\Neos\Routing\ArticleDetailRoutePart;

class ArticleDetailRoutePartTest extends FunctionalTestCase
{
    /** @var ArticleDetailRoutePart */
    private $subject;

    public function setUp()
    {
        parent::setUp();

        $this->subject = new ArticleDetailRoutePart();
        $this->inject($this->subject, 'pageFrameLinkingService', new PageFrameLinkingService());
        $this->inject(
            $this->subject,
            'types',
            [
                '_fallback' => 'poi/detail/fallback',
                '42' => 'poi/detail/for-42',
                '24' => '/has/leading/and/trailing/slashes/',
            ]
        );
    }

    public function testRoutesStartingWithTheGivenPrefixesMatch(): void
    {
        $url = 'poi/detail/fallback/my-detail-url';
        $this->assertTrue($this->subject->match($url), 'URL ' . $url . ' should match');

        $url = 'poi/detail/for-42/my-detail-url';
        $this->assertTrue($this->subject->match($url), 'URL ' . $url . ' should match');
    }

    public function testRoutesWithGivenPrefixesNotAtStartDontMatch(): void
    {
        $url = 'foo/bar/poi/detail/fallback/my-detail-url';
        $this->assertFalse($this->subject->match($url), 'URL ' . $url . ' should match');

        $url = 'some/other/path/poi/detail/for-42/my-detail-url';
        $this->assertFalse($this->subject->match($url), 'URL ' . $url . ' should match');
    }

    public function testStripsPrefixFromMatchingUrl(): void
    {
        $url = 'poi/detail/fallback/my-detail-url';
        $this->assertTrue($this->subject->match($url), 'URL ' . $url . ' should match');
        $this->assertEquals('/my-detail-url', $url);
    }

    public function testDoesNotStripAnythingFromUrlIfNotMatching(): void
    {
        $url = 'non/matching/url';
        $this->assertFalse($this->subject->match($url), 'URL ' . $url . ' should not match');
        $this->assertEquals('non/matching/url', $url);
    }

    public function testAlsoMatchesIfDefinedWithLeadingAndTrailingSlashes(): void
    {
        $url = 'has/leading/and/trailing/slashes/my-detail-url';
        $this->assertTrue($this->subject->match($url), 'URL ' . $url . ' should match');
    }

    public function testDoesNotRemoveSlashFromUrlIfDeclaredWithSurroundingSlashes(): void
    {
        $url = 'has/leading/and/trailing/slashes/my-detail-url';
        $this->assertTrue($this->subject->match($url), 'URL ' . $url . ' should match');
        $this->assertEquals('/my-detail-url', $url);
    }

    public function testDoesNotResolveUrlIfNotPointingToPageFrame(): void
    {
        $routeValues = $this->routeValues(
            [
                '@package' => 'foo.bar',
                '@controller' => 'foo',
                '@action' => 'bar',
            ]
        );
        $this->assertFalse($this->subject->resolve($routeValues));
    }

    public function testDoesNotResolveUrlIfTypeArgumentNotSpecified(): void
    {
        $routeValues = $this->routeValues();
        $this->assertFalse($this->subject->resolve($routeValues));
    }

    public function testResolvesUrlWithCorrectUrlFragmentForType(): void
    {
        $routeValues = $this->routeValues([ '--plugin' => [ '_type' => '42' ] ]);
        $result = $this->subject->resolve($routeValues);

        $this->assertNotFalse($result);
        $this->assertInstanceOf(ResolveResult::class, $result);
        $this->assertEquals('poi/detail/for-42', $result->getResolvedValue());
    }

    public function testWillRemoveTypeFromRouteValuesIfResolves(): void
    {
        $routeValues = $this->routeValues([ '--plugin' => [ '_type' => '42' ] ]);

        $this->assertNotFalse($this->subject->resolve($routeValues));
        $this->assertNull($routeValues['--plugin']['_type'] ?? null);
    }

    public function testResolvesUrlWithFallbackUrlFragmentIfTypeNotConfigured(): void
    {
        $routeValues = $this->routeValues([ '--plugin' => [ '_type' => '999' ] ]);
        $result = $this->subject->resolve($routeValues);

        $this->assertNotFalse($result);
        $this->assertInstanceOf(ResolveResult::class, $result);
        $this->assertEquals('poi/detail/fallback', $result->getResolvedValue());
    }

    /**
     * Returns a route values array that is valid for the detail view if you add `--plugin._type`.
     *
     * @param array $overrides
     * @return array
     */
    private function routeValues(array $overrides = []): array
    {
        return array_replace_recursive(
            [
                '@package' => 'newland.pageframeprovider',
                '@controller' => 'pageframe',
                '@action' => 'show',
                '--plugin' => [
                    '@package' => 'newland.toubiz.poi.neos',
                    '@controller' => 'pointofinterests',
                    '@action' => 'show',
                ],
            ],
            $overrides
        );
    }
}
