<?php
namespace Newland\Toubiz\Poi\Neos\Service;

/*
 * This file is part of the "toubiz-poi-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\ContentRepository\Domain\Model\Node;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Mvc\Controller\ControllerContext;
use Newland\NeosCommon\Service\ControllerContextFactory;
use Newland\PageFrameProvider\Service\PageFrameLinkingService;
use Newland\Toubiz\Poi\Neos\Routing\ArticleDetailRoutePart;
use Newland\Toubiz\Sync\Neos\Domain\Model\Article;

/**
 * Generates detail URLs for articles.
 *
 * This Service is tightly coupled to the `PageFrameController` functionality of `Newland.NeosCommon` to have
 * detail pages that are independent of the PageTree as well as the `ArticleTypeHandler` which handles the URLs
 * generated by this service in order to provide nice looking URLs.
 *
 * @Flow\Scope("singleton")
 */
class ArticleUrlService
{

    /**
     * @Flow\InjectConfiguration(package="Newland.Toubiz.Poi.Neos.lodging.targetUri")
     * @var string
     * @deprecated since 1.3.0, superseeded by $lodgingBaseUri
     */
    protected $lodgingUri;

    /**
     * @Flow\InjectConfiguration(package="Newland.Toubiz.Poi.Neos.lodging.targetBaseUri")
     * @var string
     */
    protected $lodgingBaseUri;

    /**
     * @var ControllerContextFactory
     * @Flow\Inject
     */
    protected $controllerContextFactory;

    /**
     * @var PageFrameLinkingService
     * @Flow\Inject()
     */
    protected $pageFrameLinkingService;

    /**
     * @param Article $article
     * @param ControllerContext $context
     * @return string|null
     * @throws \Neos\Eel\Exception
     * @throws \Neos\Neos\Exception
     */
    public function generateUrl(Article $article, ControllerContext $context)
    {
        $url = $article->getIsLodging() ? $this->generateLodgingUrl($article) : null;
        $url = $url ?? $this->generateDetailViewUrl($article, $context);

        return $url;
    }

    /**
     * Some requests are within a controller context that has no "current node".
     * So we pass the current node manually and create a fake controller context around that.
     *
     * @param Article $article
     * @param Node $node
     * @return mixed|string|null
     */
    public function generateUrlByCurrentNode(Article $article, Node $node)
    {
        $context = $this->controllerContextFactory->initializeFakeControllerContext($node);

        return $this->generateUrl($article, $context);
    }

    /**
     * @param Article $article
     * @return string|null
     */
    private function generateLodgingUrl(Article $article)
    {
        if ($this->lodgingUri) {
            return str_replace(':id', $article->getOriginalId(), $this->lodgingUri);
        }

        if ($this->lodgingBaseUri) {
            return $this->lodgingBaseUri . $article->getDetailUri();
        }

        return null;
    }

    private function generateDetailViewUrl(Article $article, ControllerContext $context): string
    {
        return $this->pageFrameLinkingService->build(
            $context->getUriBuilder(),
            ArticleDetailRoutePart::PACKAGE,
            ArticleDetailRoutePart::CONTROLLER,
            ArticleDetailRoutePart::ACTION,
            'poiDetail',
            [
                'article' => $article,
                ArticleDetailRoutePart::TYPE_ARGUMENT => $article->getMainType(),
            ],
            ArticleDetailRoutePart::ARGUMENT_NAMESPACE
        );
    }
}
