<?php

namespace Newland\Toubiz\Poi\Neos\Routing;

use Neos\Flow\Annotations as Flow;
use Neos\Flow\Mvc\Routing\AbstractRoutePart;
use Neos\Flow\Mvc\Routing\Dto\MatchResult;
use Neos\Flow\Mvc\Routing\Dto\ResolveResult;
use Neos\Flow\Mvc\Routing\DynamicRoutePartInterface;
use Newland\PageFrameProvider\Service\PageFrameLinkingService;

/**
 * Article Type Routing handler.
 *
 * Handles URLs that are built for PointOfInterests::showAction inside of a PageFrame.
 * The URLs can be configured in the `Newland.Toubiz.Poi.Neos.detailPage` path which can contain
 * the article type as a key and the URL as a value.
 *
 * This Handler is tightly coupled to `ArticleUrlService` which generates URLs handled by this Route
 * Handler.
 *
 * @Flow\Scope("prototype")
 */
class ArticleDetailRoutePart extends AbstractRoutePart implements DynamicRoutePartInterface
{
    // Constants that identify the POI plugin.
    const PACKAGE = 'newland.toubiz.poi.neos';
    const CONTROLLER = 'pointofinterests';
    const ACTION = 'show';

    // Plugin namespace as setup by the Fusion Object.
    const ARGUMENT_NAMESPACE = PageFrameLinkingService::DEFAULT_ARGUMENT_NAMESPACE;

    // The plugin argument that is used to determine the article type.
    const TYPE_ARGUMENT = '_type';

    // Configuration key to use if there is no detail URL for the specific article type.
    const TYPE_FALLBACK = '_fallback';

    // Url that is used if there is not even a `_fallback` configured (instead of simply failing altogether).
    const ULTIMATE_FALLBACK_URL = 'poi/detail';

    /**
     * @var array
     * @Flow\InjectConfiguration(package="Newland.Toubiz.Poi.Neos", path="detailPage")
     */
    protected $types;

    /**
     * @var PageFrameLinkingService
     * @Flow\Inject()
     */
    protected $pageFrameLinkingService;

    /**
     * Checks whether this Route Part corresponds to the given $routePath.
     * This method does not only check if the Route Part matches. It can also
     * shorten the $routePath by the matching substring when matching is successful.
     * This is why $routePath has to be passed by reference.
     *
     * @param string &$routePath The request path to be matched - without query parameters, host and fragment.
     * @return bool|MatchResult true or an instance of MatchResult if Route Part matched $routePath, otherwise false.
     */
    public function match(&$routePath)
    {
        $matchGroup = implode(
            '|',
            array_map(
                function (string $part) {
                    return preg_quote($part, '/');
                },
                array_values($this->types())
            )
        );

        $regex = sprintf('/^(%s)/', $matchGroup);
        if (preg_match($regex, $routePath)) {
            $routePath = preg_replace($regex, '', $routePath);
            return true;
        }

        return false;
    }

    /**
     * Checks whether this Route Part corresponds to the given $routeValues.
     * This method does not only check if the Route Part matches. It also
     * removes resolved elements from $routeValues-Array.
     * This is why $routeValues has to be passed by reference.
     *
     * @param array &$routeValues An array with key/value pairs to be resolved by Dynamic Route Parts.
     * @return bool|ResolveResult true or an instance of ResolveResult if Route Part can resolve one or more
     *     $routeValues elements, otherwise false.
     */
    public function resolve(array &$routeValues)
    {
        if (!$this->isTryingToBuildArticleUrl($routeValues)) {
            return false;
        }

        $type = $routeValues[static::ARGUMENT_NAMESPACE][static::TYPE_ARGUMENT] ?? static::TYPE_FALLBACK;
        unset($routeValues[static::ARGUMENT_NAMESPACE][static::TYPE_ARGUMENT]);
        return new ResolveResult($this->urlFragmentForType($type));
    }

    /**
     * @param array $routeValues
     * @return bool
     */
    private function isTryingToBuildArticleUrl(array $routeValues): bool
    {
        $type = $routeValues[static::ARGUMENT_NAMESPACE][static::TYPE_ARGUMENT] ?? null;

        $isPageFrameActionLink = $this->pageFrameLinkingService->isPageFrameActionLink(
            $routeValues,
            static::PACKAGE,
            static::CONTROLLER,
            static::ACTION,
            static::ARGUMENT_NAMESPACE
        );

        return $isPageFrameActionLink && $type !== null;
    }

    /**
     * @param string|null $type
     * @return string
     */
    private function urlFragmentForType(string $type = null): string
    {
        $types = $this->types();

        // Use `_fallback` configuration if there is no URL for the type.
        if ($type === null || !array_key_exists($type, $types)) {
            $type = static::TYPE_FALLBACK;
        }

        // Use ultimate fallback if not even `_fallback` is configured.
        if (!array_key_exists($type, $types)) {
            return static::ULTIMATE_FALLBACK_URL;
        }

        return trim($types[$type], '/');
    }


    /**
     * Prepares the URLs given by the configuration for usage in this RoutePart.
     * To ensure that further routing works as expected it is important that no leading
     * or trailing slashes are included in the configuration.
     *
     * @return string[]
     */
    private function types(): array
    {
        if (!$this->types) {
            return [];
        }

        return array_map(
            function (string $url) {
                return trim($url, '/');
            },
            $this->types
        );
    }

    public function setSplitString($splitString)
    {
        // Does nothing.
    }
}
