<?php declare(strict_types=1);

namespace Newland\Toubiz\Poi\Neos;

use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Common\Neos\Translation\TranslatesNodeProperties;
use Newland\Toubiz\Sync\Neos\Domain\Model\Article;
use Newland\Toubiz\Sync\Neos\Domain\Model\Category;
use Newland\Toubiz\Sync\Neos\Domain\Repository\ArticleRepository;
use Newland\Toubiz\Sync\Neos\Domain\Repository\CategoryRepository;

class TranslateNodeProperties
{

    /**
     * @var ArticleRepository
     * @Flow\Inject()
     */
    protected $articleRepository;

    /**
     * @var CategoryRepository
     * @Flow\Inject()
     */
    protected $categoryRepository;

    /**
     * @TranslatesNodeProperties(
     *     nodeTypes={
     *          "Newland.Toubiz.Poi.Neos:Teaser",
     *          "Newland.Toubiz.Poi.Neos:Show",
     *      }
     * )
     */
    public function translateSingleReferences(array $properties, string $targetLanguage): array
    {
        $translated = $this->translateArticle($properties['article'] ?? null, $targetLanguage);
        if ($translated) {
            $properties['article'] = $translated;
        } else {
            unset($properties['article']);
        }

        return $properties;
    }

    /**
     * @TranslatesNodeProperties(
     *     nodeTypes={
     *          "Newland.Toubiz.Poi.Neos:List",
     *          "Newland.Toubiz.Poi.Neos:FilteredLists",
     *      }
     * )
     */
    public function translatePreselectedCities(array $properties, string $targetLanguage): array
    {
        $properties['preselectedCities'] = array_values(array_filter(array_map(
            function (string $id) use ($targetLanguage) {
                return $this->translateArticle($id, $targetLanguage);
            },
            $properties['preselectedCities'] ?? []
        )));

        return $properties;
    }

    /**
     * @TranslatesNodeProperties(
     *     nodeTypes={
     *          "Newland.Toubiz.Poi.Neos:List",
     *          "Newland.Toubiz.Poi.Neos:FilteredLists",
     *      }
     * )
     */
    public function translatePreselectedCategories(array $properties, string $targetLanguage): array
    {
        $properties['preselectedCategories'] = array_values(array_filter(array_map(
            function (string $id) use ($targetLanguage) {
                return $this->translateCategory($id, $targetLanguage);
            },
            $properties['preselectedCategories'] ?? []
        )));

        return $properties;
    }

    private function translateArticle(
        ?string $id,
        string $targetLanguage
    ): ?string {
        if ($id === null) {
            return null;
        }
        $article = $this->articleRepository->withoutLanguageHandling(function () use ($id) {
            return $this->articleRepository->findByIdentifier($id);
        });

        if (!$article) {
            return null;
        }

        $languageVariant = $this->articleRepository->translateInto($article, $targetLanguage);

        if ($languageVariant instanceof Article) {
            return $languageVariant->getPersistenceObjectIdentifier();
        }

        return null;
    }

    private function translateCategory(
        string $id,
        string $targetLanguage
    ): ?string {
        $category = $this->categoryRepository->withoutLanguageHandling(function () use ($id) {
            return $this->categoryRepository->findByIdentifier($id);
        });

        if (!$category) {
            return null;
        }

        $languageVariant = $this->categoryRepository->translateInto($category, $targetLanguage);

        if ($languageVariant instanceof Category) {
            return $languageVariant->getPersistenceObjectIdentifier();
        }

        return null;
    }
}
