<?php

namespace Newland\Toubiz\Poi\Neos\DataSource;

/*
 * This file is part of the "toubiz-poi-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\I18n\Translator;
use Neos\Neos\Service\DataSource\AbstractDataSource;
use Newland\NeosCommon\Domain\Model\Topic;
use Newland\Toubiz\Poi\Neos\Domain\Repository\TopicRepository;

/**
 * Categories data source service.
 *
 * @Flow\Scope("singleton")
 */
class TopicsDataSource extends AbstractDataSource
{
    /**
     * Data source identifier.
     *
     * @var string
     */
    protected static $identifier = 'newland-toubiz-poi-neos-topics';

    /**
     * @var Translator
     * @Flow\Inject
     */
    protected $translator;

    /**
     * @var TopicRepository
     * @Flow\Inject
     */
    protected $topicRepository;

    /**
     * Get requested data from source.
     *
     * The return value must be JSON serializable data structure.
     *
     * @api
     * @param NodeInterface $node The node that is currently edited (optional)
     * @param array $arguments Additional arguments (key / value)
     * @return mixed JSON serializable data
     * @throws \Neos\ContentRepository\Exception\NodeException
     */
    public function getData(NodeInterface $node = null, array $arguments = [])
    {
        $groups = [
            TopicRepository::TYPE_ATTRIBUTE => $this->translate('type.attributes'),
            TopicRepository::TYPE_CATEGORY => $this->translate('type.categories'),
        ];

        $articleType = null;
        if ($node && $node->hasProperty('articleType')) {
            $articleType = $node->getProperty('articleType');
        }

        $language = $this->getLanguage($node);
        if ($language !== null) {
            $this->topicRepository->setLanguage($language);
        }

        /*
         * Added for legacy support of older teaser lists
         * Previous teaser list would not scope by articleType,
         * so the decision was not to return properties before the legacy node is removed
        */

        if ($articleType === null) {
            $topics = $this->topicRepository->findAllCategory([ 'articleType' => $articleType ]);
        } else {
            $topics = $this->topicRepository->findAll([ 'articleType' => $articleType ]);
        }

        $data = [];
        $ids = [];
        foreach ($topics as $topic) {
            $ids[$topic->getCombinedIdentifier()] = true;
            $data[] = [
                'label' => $topic->getTitle(),
                'value' => $topic->getCombinedIdentifier(),
                'group' => $groups[$topic->getType()],
            ];
        }

        if ($node !== null) {
            $data = $this->addUnknownPreselectedTopics($data, $node, $ids);
        }

        return $data;
    }

    private function addUnknownPreselectedTopics(array $data, NodeInterface $node, array $ids): array
    {
        foreach ($node->getProperty('preselectedTopics') ?? [] as $preselected) {
            if (!array_key_exists($preselected, $ids)) {
                $data[] = [
                    'value' => $preselected,
                    'label' => $this->translate('attribute.unknown') . ' ' . $preselected,
                    'group' => $this->translate('type.unknown'),
                ];
            }
        }

        return $data;
    }

    /**
     * @param string $id
     * @return string|null
     */
    private function translate(string $id)
    {
        return $this->translator->translateById(
            $id,
            [],
            null,
            null,
            'Models/Topic',
            'Newland.Toubiz.Poi.Neos'
        );
    }

    private function getLanguage(NodeInterface $node = null): ?string
    {
        if ($node === null) {
            return null;
        }

        $dimensions = $node->getDimensions();

        return $dimensions['language'][0] ?? null;
    }
}
