<?php

namespace Newland\Toubiz\Poi\Neos\Domain\Repository;

use Newland\NeosCommon\Domain\Repository\TopicRepository as CommonTopicRepository;
use Newland\NeosCommon\Domain\Model\Topic;
use Newland\Toubiz\Poi\Neos\Service\AttributeTopicService;
use Newland\Toubiz\Sync\Neos\Domain\Filter\CategoryFilter;
use Newland\Toubiz\Sync\Neos\Domain\Model\Category;
use Newland\Toubiz\Sync\Neos\Domain\Repository\CategoryRepository;
use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Sync\Neos\Translation\TranslatableRepository;

/**
 * @Flow\Scope("singleton")
 */
class TopicRepository extends CommonTopicRepository
{
    use TranslatableRepository;

    const TYPE_CATEGORY = 'category';
    const TYPE_ATTRIBUTE = 'attribute';

    /**
     * Types of topics this repository handles.
     * For all types in this array there must be a `findAll{Type}` and `findBy{Type}`
     * method.
     *
     * @var string[]
     */
    protected $types = [
        self::TYPE_CATEGORY,
        self::TYPE_ATTRIBUTE,
    ];

    /**
     * @var CategoryRepository
     * @Flow\Inject()
     */
    protected $categoryRepository;

    /**
     * @var AttributeTopicService
     * @Flow\Inject()
     */
    protected $attributeTopicService;

    /**
     * @param array $config
     * @return Topic[]
     */
    public function findAllCategory(array $config = []): iterable
    {
        $articleType = $config['articleType'] ?? null;
        $filter = new CategoryFilter();
        $filter->setOrderBy([ 'category.title' => 'ASC' ]);

        if ($articleType !== null) {
            $filter->setArticleMainType($articleType);
        }

        return array_map(
            function (Category $category) {
                return $this->topicFromCategory($category);
            },
            $this->categoryRepository->findByFilter($filter)
        );
    }

    /**
     * @param string $categoryId
     * @return Topic|null
     */
    public function findByCategory(string $categoryId)
    {
        $category = $this->categoryRepository->findByIdentifier($categoryId);
        if (!$category) {
            return null;
        }

        return $this->topicFromCategory($category);
    }

    /**
     * @param array $config
     * @return Topic[]
     */
    public function findAllAttribute(array $config = []): iterable
    {
        $articleType = $config['articleType'] ?? null;
        return $this->findByIdentifiers(
            $this->attributeTopicService->getConfiguredIdentifiers($articleType),
            true
        );
    }

    /**
     * @param string $attributeString
     * @return Topic|null
     */
    public function findByAttribute(string $attributeString)
    {
        $exploded = explode(':', $attributeString);
        $attribute = $exploded[0];
        $value = \count($exploded) > 1 ? $exploded[1] : null;

        $title = $this->attributeTopicService->titleForAttribute($attribute, $value) ?: $attributeString;
        return new Topic(
            self::TYPE_ATTRIBUTE,
            $attribute,
            $title,
            $value
        );
    }


    private function topicFromCategory(Category $category): Topic
    {
        return new Topic(
            self::TYPE_CATEGORY,
            $category->getPersistenceObjectIdentifier(),
            $category->getTitle()
        );
    }
}
