<?php

namespace Newland\Toubiz\Poi\Neos\Domain\Repository;

use Neos\Flow\ObjectManagement\ObjectManager;
use Neos\Utility\Arrays;
use Newland\Toubiz\Poi\Neos\Domain\Model\Topic;
use Newland\Toubiz\Poi\Neos\Service\AttributeTopicService;
use Newland\Toubiz\Sync\Neos\Domain\Filter\ArticleFilter;
use Newland\Toubiz\Sync\Neos\Domain\Filter\CategoryFilter;
use Newland\Toubiz\Sync\Neos\Domain\Model\Category;
use Newland\Toubiz\Sync\Neos\Domain\Repository\ArticleRepository;
use Newland\Toubiz\Sync\Neos\Domain\Repository\CategoryRepository;
use Neos\Flow\Annotations as Flow;

/**
 * @Flow\Scope("singleton")
 */
class TopicRepository
{
    /**
     * @var CategoryRepository
     * @Flow\Inject()
     */
    protected $categoryRepository;

    /**
     * @var ArticleRepository
     * @Flow\Inject()
     */
    protected $articleRepository;

    /**
     * @var ObjectManager
     * @Flow\Inject
     */
    protected $objectManager;

    /**
     * @var AttributeTopicService
     * @Flow\Inject()
     */
    protected $attributeTopicService;

    /**
     * @var int
     */
    private $limit = 10;

    public function setLimit(int $limit)
    {
        $this->limit = $limit;
    }

    /**
     * @param string|int|null $articleType
     * @return Topic[]
     */
    public function findAll($articleType = null): iterable
    {

        return $this->findByArticleType($articleType);
    }

    /**
     * @param string|int|null $articleType
     * @return Topic[]
     */
    private function findByArticleType($articleType): array
    {
        $topics = array_merge(
            $this->findAllCategories($articleType),
            $this->findAllAttributes($articleType)
        );

        return $this->sortTopics($topics);
    }

    /**
     * @param string|int|null $articleType
     * @return Topic[]
     */
    public function findAllCategories($articleType = null): iterable
    {
        $filter = new CategoryFilter();
        $filter->setOrderBy([ 'category.title' => 'ASC' ]);

        if ($articleType !== null) {
            $filter->setArticleMainType($articleType);
        }

        return array_map(
            function (Category $category) {
                return $this->topicFromCategory($category);
            },
            $this->categoryRepository->findByFilter($filter)
        );
    }

    /**
     * @param string|int|null $articleType
     * @return Topic[]
     */
    public function findAllAttributes($articleType = null): iterable
    {
        return $this->findByIdentifiers(
            $this->attributeTopicService->getConfiguredIdentifiers($articleType),
            true
        );
    }

    /**
     * @param array $identifiers
     * @param bool $sort
     * @return Topic[]
     */
    public function findByIdentifiers(array $identifiers, bool $sort = false): iterable
    {
        $topics = [];

        foreach ($identifiers as $identifier) {
            $topic = $this->findByIdentifier($identifier);
            if ($topic) {
                $topics[] = $topic;
            }
        }
        if ($sort) {
            $topics = $this->sortTopics($topics);
        }
        return $topics;
    }

    /**
     * @param string $identifier
     * @return Topic|null
     */
    public function findByIdentifier(string $identifier)
    {
        if (strpos($identifier, ':') === false) {
            return null;
        }
        list($type, $id) = explode(':', $identifier, 2);

        switch ($type) {
            case Topic::TYPE_CATEGORY:
                return $this->findByCategory($id);

            case Topic::TYPE_ATTRIBUTE:
                return $this->findByAttribute($id);
        }

        return null;
    }

    /**
     * @param string $attributeString
     * @return Topic|null
     */
    public function findByAttribute(string $attributeString)
    {
        $exploded = explode(':', $attributeString);
        $attribute = $exploded[0];
        $value = \count($exploded) > 1 ? $exploded[1] : null;

        $filter = new ArticleFilter();
        $filter->setLimit($this->limit);
        $filter->setAttributesIn(
            [
                $attribute => [ $value === true ? 'true' : $value ],
            ]
        );

        $title = $this->attributeTopicService->titleForAttribute($attribute, $value) ?: $attributeString;
        return (new Topic())
            ->setType(Topic::TYPE_ATTRIBUTE)
            ->setIdentifier($attribute)
            ->setCombinedIdentifier(Topic::TYPE_ATTRIBUTE . ':' . $attributeString)
            ->setTitle($title)
            ->setArticleGetter(
                function () use ($filter) {
                    return $this->articleRepository->findByFilter($filter);
                }
            )
            ->setTotalCount($this->articleRepository->countByFilter($filter, $this->limit)['items'])
            ->setValue($value);
    }

    /**
     * @param string $categoryId
     * @return Topic|null
     */
    public function findByCategory(string $categoryId)
    {
        $category = $this->categoryRepository->findByIdentifier($categoryId);
        if (!$category) {
            return null;
        }

        return $this->topicFromCategory($category);
    }

    public function topicFromCategory(Category $category): Topic
    {
        $filter = new ArticleFilter();
        $filter->setCategories([ $category->getPersistenceObjectIdentifier() ]);
        $filter->setLimit($this->limit);

        return (new Topic())
            ->setType(Topic::TYPE_CATEGORY)
            ->setIdentifier($category->getPersistenceObjectIdentifier())
            ->setCombinedIdentifier(Topic::TYPE_CATEGORY . ':' . $category->getPersistenceObjectIdentifier())
            ->setTitle($category->getTitle())
            ->setValue($category->getPersistenceObjectIdentifier())
            ->setTotalCount($this->articleRepository->countByFilter($filter, $this->limit)['items'])
            ->setArticleGetter(
                function () use ($filter) {
                    return $this->articleRepository->findByFilter($filter);
                }
            );
    }

    /**
     * @param array $topics
     * @return array
     */
    private function sortTopics(array $topics)
    {
        usort(
            $topics,
            function ($topic1, $topic2) {
                return strnatcasecmp($topic1->getTitle(), $topic2->getTitle());
            }
        );
        return $topics;
    }
}
