<?php
namespace Newland\Toubiz\Poi\Neos\Service;

/*
 * This file is part of the "toubiz-poi-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\ContentRepository\Domain\Model\Node;
use Neos\Eel\FlowQuery\FlowQuery;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Mvc\Controller\ControllerContext;
use Neos\Neos\Service\LinkingService;
use Newland\NeosCommon\Service\ControllerContextFactory;
use Newland\Toubiz\Sync\Neos\Domain\Model\Article;

/**
 * @Flow\Scope("singleton")
 */
class ArticleUrlService
{
    /**
     * @var Node[]
     */
    private $nodeCache = [];

    /**
     * @Flow\InjectConfiguration(package="Newland.Toubiz.Poi.Neos.lodging.targetUri")
     * @var string
     * @deprecated since 1.3.0, superseeded by $lodgingBaseUri
     */
    protected $lodgingUri;

    /**
     * @Flow\InjectConfiguration(package="Newland.Toubiz.Poi.Neos.lodging.targetBaseUri")
     * @var string
     */
    protected $lodgingBaseUri;

    /**
     * @Flow\Inject
     * @var LinkingService
     */
    protected $linkingService;

    /**
     * @var ControllerContextFactory
     * @Flow\Inject
     */
    protected $controllerContextFactory;

    /**
     * @param Article $article
     * @param ControllerContext $context
     * @return mixed|null|string
     * @throws \Neos\Eel\Exception
     * @throws \Neos\Neos\Exception
     */
    public function generateUrl(Article $article, ControllerContext $context)
    {
        $url = $article->getIsLodging() ? $this->generateLodgingUrl($article) : null;
        $url = $url ?? $this->generateNodeBasedUrl($article, $context);

        return $url;
    }

    /**
     * Some requests are within a controller context that has no "current node".
     * So we pass the current node manually and create a fake controller context around that.
     *
     * @param Article $article
     * @param Node $node
     * @return mixed|string|null
     */
    public function generateUrlByCurrentNode(Article $article, Node $node)
    {
        $context = $this->controllerContextFactory->initializeFakeControllerContext($node);

        return $this->generateUrl($article, $context);
    }

    /**
     * @param Article $article
     * @param ControllerContext $context
     * @return null|string
     * @throws \Neos\Eel\Exception
     * @throws \Neos\Neos\Exception
     */
    public function generateNodeBasedUrl(Article $article, ControllerContext $context)
    {
        $node = $this->nodeForArticleFromControllerContext($article, $context);
        if (!$node) {
            return null;
        }

        return $this->linkingService->createNodeUri(
            $context,
            $node,
            null,
            null,
            true,
            [
                '--newland_toubiz_poi_neos-list' => [
                    '@package' => 'newland.toubiz.poi.neos',
                    '@controller' => 'pointofinterests',
                    '@action' => 'show',
                    'article' => $article,
                ],
            ]
        );
    }

    /**
     * @param Article $article
     * @return string|null
     */
    public function generateLodgingUrl(Article $article)
    {
        if ($this->lodgingUri) {
            return str_replace(':id', $article->getOriginalId(), $this->lodgingUri);
        }

        if ($this->lodgingBaseUri) {
            return $this->lodgingBaseUri . $article->getDetailUri();
        }
    }

    /**
     * @param Article $article
     * @param ControllerContext $context
     * @return Node|null
     * @throws \Neos\Eel\Exception
     */
    public function nodeForArticleFromControllerContext(Article $article, ControllerContext $context)
    {
        return $this->nodeForArticle(
            $article,
            $context
                ->getRequest()
                ->getInternalArgument('__node')
                ->getContext()
                ->getCurrentSiteNode()
        );
    }

    /**
     * @param Article $article
     * @param Node $site
     * @return null|Node
     * @throws \Neos\Eel\Exception
     */
    public function nodeForArticle(Article $article, Node $site)
    {
        $key = $site->getPath() . $article->getMainType();
        if (!array_key_exists($key, $this->nodeCache)) {
            $this->nodeCache[$key] = $this->findNode($article->getMainType(), $site);
        }
        return $this->nodeCache[$key];
    }

    /**
     * @param int $articleType
     * @param Node $site
     * @return Node|null
     * @throws \Neos\Eel\Exception
     */
    private function findNode(int $articleType, Node $site)
    {
        $selector = implode(
            ',',
            [
                '[instanceof Newland.Toubiz.Poi.Neos:List][isMainView=true][articleType *= "' . $articleType . '"]',
                '[instanceof Newland.Toubiz.Poi.Neos:List][articleType *= "' . $articleType . '"]',
                '[instanceof Newland.Toubiz.Poi.Neos:List]',
            ]
        );

        return (new FlowQuery([ $site ]))->find($selector)->parents('[instanceof Neos.NodeTypes:Page]')->get(0);
    }
}
