<?php
namespace Newland\Toubiz\Poi\Neos\Controller;

/*
 * This file is part of the "toubiz-poi-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Neos\Flow\Http\Exception as HttpException;
use Neos\Flow\Mvc\Controller\MvcPropertyMappingConfiguration;
use Newland\PageFrameProvider\Service\PageFrameLinkingService;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Poi\Neos\Service\ArticleFilterService;
use Newland\Toubiz\Poi\Neos\Service\ArticleUrlService;
use Newland\Toubiz\Sync\Neos\Domain\Filter\ArticleFilter;
use Newland\Toubiz\Sync\Neos\Domain\Model\Article;

class PointOfInterestsController extends AbstractActionController
{

    /**
     * @var ArticleFilterService
     * @Flow\Inject()
     */
    protected $articleFilterService;

    /**
     * @var ArticleUrlService
     * @Flow\Inject()
     */
    protected $articleUrlService;

    /**
     * Teaser action.
     *
     * Renders a single teaser.
     *
     * @return void
     */
    public function teaserAction(): void
    {
        if (array_key_exists('article', $this->properties)) {
            $article = $this->articleRepository->find($this->properties['article']);
            if ($article) {
                $this->view->assign('article', $article);
            }
        }
        $this->view->assignMultiple(
            [
                'wrapStart' => $this->request->getInternalArgument('__wrapStart'),
                'wrapEnd' => $this->request->getInternalArgument('__wrapEnd'),
            ]
        );
    }

    /**
     * Renders the `Newland.Toubiz.Poi.Neos:TeaserList` NodeType.
     *
     * @return void
     * @deprecated since 1.13, will be disabled by default in 1.13 and removed in 1.15
     *
     */
    public function teaserListAction(): void
    {
        $filter = $this->createArticleFilter();
        $articles = $this->articleRepository->findByFilter($filter);
        $displayStyle = $this->properties['displayStyle'] ?? '';
        $this->view->assign('articles', $articles);
        $this->view->assign('enableSwiper', $displayStyle === 'swiper');
    }

    public function showAction(Article $article = null): void
    {
        $this->redirectToStaticDetailPageIfNotOnIt($article);

        $insertGalleryAfterLinks = false;

        if ($this->properties['article'] ?? null) {
            $article = $this->articleRepository->find($this->properties['article']);

            if ($article) {
                $insertGalleryAfterLinks =
                    $article->getMainType() === ArticleConstants::TYPE_CITY &&
                    ($this->settings['cityDetails']['gallery']['insertAfterLinks'] ?? false);
            }
        }

        $this->view->assignMultiple(
            [
                'url' => (string) $this->request->getHttpRequest()->getUri(),
                'article' => $article,
                'similarTours' => $this->getSimilarToursIfApplicable($article),
                'insertGalleryAfterLinks' => $insertGalleryAfterLinks,
            ]
        );
    }

    /**
     * Creates a new article filter based on the given properties array
     * (or the properties of the current node if no argument is provided).
     *
     * @param array|null $properties
     * @return ArticleFilter
     */
    private function createArticleFilter(array $properties = null): ArticleFilter
    {
        $properties = $properties ?: $this->properties;

        $filter = new ArticleFilter();
        $filter->setCurrentSite($this->currentSite->getNodeName());

        if (array_key_exists('topicFilter', $properties)) {
            $filter = $this->articleFilterService->setFilterFromTopics($properties['topicFilter'], $filter);
        } elseif (array_key_exists('categoryFilter', $properties)) {
            $filter = $this->articleFilterService->setFilterFromTopics($properties['categoryFilter'], $filter);
        } elseif (array_key_exists('categories', $properties) && is_array($properties['categories'])) {
            $filter->setCategories($properties['categories']);
        }

        if (array_key_exists('recordLimit', $properties)) {
            $filter->setLimit($properties['recordLimit']);
        }

        if (array_key_exists('articleType', $properties)) {
            $filter->setMainType((int) $properties['articleType']);
        }

        /**
         * @deprecated
         * The "cities" attribute cannot be edited anymore but might still be
         * configured. Use "regions" instead.
         */
        if (array_key_exists('cities', $properties)) {
            $filter->setZips($this->properties['cities']);
        }

        if (array_key_exists('regions', $properties)) {
            $filter->setZips($this->regionService->getZipsFromRegionKeys($properties['regions']));
        }

        return $filter;
    }


    /**
     * @param Article|null $article
     * @return void
     */
    private function redirectToStaticDetailPageIfNotOnIt(Article $article = null): void
    {
        if ($article === null) {
            return;
        }

        if (strtolower($this->request->getMainRequest()->getControllerName()) !== PageFrameLinkingService::CONTROLLER) {
            $this->redirectToUri(
                $this->articleUrlService->generateUrl($article, $this->controllerContext),
                0,
                301
            );
        }
    }

    private function getSimilarToursIfApplicable(?Article $article): ?array
    {
        if ($article === null) {
            return null;
        }

        if ($article->getMainType() !== ArticleConstants::TYPE_TOUR) {
            return null;
        }

        $filter = new ArticleFilter();
        $filter->setCurrentSite($this->currentSite->getNodeName());
        $filter->setCategoryCollection($article->getCategories())
            ->setExcludes([ $article ])
            ->setLimit(4);

        return (array) $this->articleRepository->findByFilter($filter);
    }
}
