<?php declare(strict_types=1);

namespace Newland\Toubiz\Poi\Neos\ViewHelpers;

use Neos\Flow\Mvc\ActionRequest;
use Neos\FluidAdaptor\Core\ViewHelper\AbstractTagBasedViewHelper;
use Neos\Flow\Annotations as Flow;
use Newland\NeosIcons\IconUrlService;
use Newland\Toubiz\Poi\Neos\Service\MapCompatibility\ApiKeyService;
use function Safe\json_encode;

class StaticMapViewHelper extends AbstractTagBasedViewHelper
{

    /**
     * @var array
     * @Flow\InjectConfiguration(package="Newland.Toubiz.Poi.Neos", path="staticMap")
     */
    protected $staticMapConfiguration;

    /**
     * @var ApiKeyService
     * @Flow\Inject()
     */
    protected $apiKeyService;

    /**
     * @var IconUrlService
     * @Flow\Inject()
     */
    protected $iconUrlService;

    public function render(
        float $latitude,
        float $longitude,
        int $width = null,
        int $height = null,
        int $zoom = null,
        string $mapType = null,
        ?array $geometry = null
    ): string {
        $mapType = $mapType ?? $this->staticMapConfiguration['mapType'] ?? null;
        $width = $width ?? $this->staticMapConfiguration['width'] ?? 800;
        $height = $height ?? $this->staticMapConfiguration['height'] ?? 600;
        $zoom = $zoom ?? $this->staticMapConfiguration['zoom'] ?? 75;

        if ($mapType === null) {
            return '<!-- Static map not rendered: No map type defined. -->';
        }

        /** @var ActionRequest $request */
        $request = $this->controllerContext->getRequest();
        $iconUrl = $this->iconUrlService->buildMarkerIconUrl(
            $this->staticMapConfiguration['style'] ?? [],
            $request
        );
        $this->addApiKeysToTags();
        $this->tag->addAttributes(
            [
                'class' => 'cursor-pointer',
                'data-toubiz-static-map' => '',
                'data-toubiz-static-map.map-type' => $mapType,
                'data-toubiz-static-map.map.zoom' => $zoom,
                'data-toubiz-static-map.width' => $width,
                'data-toubiz-static-map.height' => $height,
                'data-toubiz-static-map.markers.0.icon-url' => $iconUrl,
                'data-toubiz-static-map.markers.0.latitude' => $latitude,
                'data-toubiz-static-map.markers.0.longitude' => $longitude,
            ]
        );

        if ($this->staticMapConfiguration['markerIcon'] ?? null) {
            $this->tag->addAttribute(
                'data-toubiz-static-map.map.static-marker-icon-url',
                $this->staticMapConfiguration['markerIcon']
            );
        }

        if ($geometry) {
            $this->tag->addAttribute('data-toubiz-static-map.markers.0.tour', json_encode($geometry));
        }

        $this->tag->forceClosingTag(true);
        return $this->tag->render();
    }

    private function addApiKeysToTags(): void
    {
        $apiKeys = $this->apiKeyService->getApiKeys($this->controllerContext->getRequest());

        if (array_key_exists('googleMaps', $apiKeys)) {
            $this->tag->addAttribute(
                'data-toubiz-static-map.map.google-maps-api-key',
                $apiKeys['googleMaps']
            );
        }

        if (array_key_exists('toursprung', $apiKeys)) {
            $this->tag->addAttribute(
                'data-toubiz-static-map.map.toursprung-api-key',
                $apiKeys['toursprung']
            );
        }
    }
}
