<?php declare(strict_types=1);

namespace Newland\Toubiz\Poi\Neos\Tests\Integration\Detail;

use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\GastronomyApiService\GastronomyAdapter;
use Newland\Toubiz\Sync\Neos\Importer\ArticleImporter;

class GastronomyDetailTest extends AbstractDetailTestCase
{
    protected function articleType(): int
    {
        return ArticleConstants::TYPE_GASTRONOMY;
    }

    /** @dataProvider provideAwardNames */
    public function testRendersAwards(string $awardName): void
    {
        $label = sprintf('___LABEL___%s___', md5(random_bytes(32)));
        $value = sprintf('___VALUE___%s___', md5(random_bytes(32)));

        // This test may change in the future, when data serialization changes
        $article = $this->article([
            'attributes' => [ $this->attributeFactory->create([
                'name' => $awardName,
                'data' => [ 'label' => $label, 'value' => $value ]
            ]) ]
        ]);

        $response = $this->requestArticleDetailPage($article);
        $this->assertResponseOk($response);
        $this->assertResponseContains($awardName, $response);
        $this->assertResponseContains($label, $response);
        $this->assertResponseContains($value, $response);
    }

    /** @dataProvider provideAwardNames */
    public function testRendersAwardsAsImportedFromToubizLegacyApi(string $awardName, string $toubizLegacyAwardName): void
    {
        // End-to-end version of the test above that should not be subject to change in serialization
        // since the serialization logic in sync and template should always change together.

        $label = sprintf('___LABEL___%s___', md5(random_bytes(32)));
        $value = sprintf('___VALUE___%s___', md5(random_bytes(32)));

        $data = json_decode(file_get_contents(__DIR__ . '/../../Fixtures/ExampleToubizLegacyGastronomyWithoutAwards.json'), true);
        $data['decoration'][$toubizLegacyAwardName] = [
            'name' => $toubizLegacyAwardName,
            'properties' => [
                'property' => md5(random_bytes(32)),
                'group' => $toubizLegacyAwardName,
                'text' => $label . '###' . md5(random_bytes(32)),
                'value' => $value . '###'
            ]
        ];
        $adapter = new GastronomyAdapter($data);

        $article = $this->objectManager->get(ArticleImporter::class)->import($adapter);

        $response = $this->requestArticleDetailPage($article);
        $this->assertResponseOk($response);
        $this->assertResponseContains($awardName, $response);
        $this->assertResponseContains($label, $response);
        $this->assertResponseContains($value, $response);
    }

    public function provideAwardNames(): array
    {
        return [
            [ 'houseAwardsGeneral', 'house_decoration' ],
            [ 'houseAwardsAral', 'house_decoration_aral' ],
            [ 'houseAwardsDehoga', 'house_decoration_dehoga' ],
            [ 'houseAwardsFine', 'house_decoration_fine' ],
            [ 'houseAwardsMichelin', 'house_decoration_michelin' ],
            [ 'houseAwardsMillau', 'house_decoration_millau' ],
            [ 'houseAwardsVarta', 'house_decoration_varta' ],
        ];
    }
}
