<?php declare(strict_types = 1);


namespace Newland\Toubiz\Poi\Neos\Tests\Unit\Encoder;


use Newland\Toubiz\Poi\Neos\Encoder\TopicsToQueryEncoder;
use PHPUnit\Framework\TestCase;

class TopicsToQueryEncoderTest extends TestCase
{

    /** @var TopicsToQueryEncoder */
    private $subject;

    public function setUp(): void
    {
        $this->subject = new TopicsToQueryEncoder();
    }

    public function testEncodesCategories(): void
    {
        $result = $this->subject->encode([ 'category:abc-def', 'category:123-456' ]);
        $this->assertArraySubset(
            [
                'categories' => [
                    'abc-def' => 'abc-def',
                    '123-456' => '123-456'
                ]
            ],
            $result
        );
    }

    /**
     * @dataProvider provideIdentifierValues
     * @param string $value
     */
    public function testEncodesAttributes(string $value): void
    {
        $result = $this->subject->encode(
            [
                'attribute:some_attribute:' . $value,
                'attribute:some_other_attribute:' . $value
            ]
        );
        $this->assertArraySubset(
            [
                'attributes' => [
                    'some_attribute' => [
                        'in' => [ $value ]
                    ],
                    'some_other_attribute' => [
                        'in' => [ $value ]
                    ]
                ]
            ],
            $result
        );
    }

    /**
     * @dataProvider provideIdentifierValues
     * @param string $tag
     */
    public function testEncodesTags(string $tag): void
    {
        $result = $this->subject->encode([ 'tag:' . $tag ]);
        $this->assertArraySubset(
            [
                'attributes' => [
                    'tag' => [
                        'in' => [ $tag ]
                    ]
                ]
            ],
            $result
        );
    }


    public function provideIdentifierValues(): array
    {

        return [
            // Synthetic samples
            [ 'foo' ],
            [ 'bar' ],
            [ 'baz' ],
            [ 'foo bar' ],

            // Common german samples
            [ 'für_rollstuhlfahrer' ],
            [ 'hier steppt der bär' ],
            [ 'Neu: Sonnenuntergang Deluxe' ],

            // Samples of other special chars of other languages
            [ 'Sîne klâwen durh die wolken sint geslagen' ],
            [ 'He wes Leovenaðes sone -- liðe him be Drihten' ],

            // Why? UTF-8, that's why.
            [ 'ᚠᛇᚻ᛫ᛒᛦᚦ᛫ᚠᚱᚩᚠᚢᚱ' ],
            [ 'Τη γλώσσα μου έδωσαν' ],
            [ 'ვეპხის ტყაოსანი შოთა' ],
            [ 'யாமறிந்த மொழிகளிலே தமிழ்மொழி ' ],
            [ 'ಬಾ ಇಲ್ಲಿ ಸಂಭವಿಸು ಇಂದೆನ್ನ ಹೃದಯ' ],
            [ ' ﻿काचं शक्नोम्यत्तुम् । नोपहिनस्ति माम् ॥' ],
            [ '⠊⠀⠉⠁⠝⠀⠑⠁⠞⠀⠛⠇⠁⠎⠎⠀⠁⠝⠙⠀⠊⠞⠀⠙⠕⠑⠎⠝⠞⠀⠓⠥⠗⠞⠀⠍⠑ ' ],
            [ '些 ࣎ 世 咹 水 晶 ও 空 ࣎ 害 咦 ' ],
            [ 'म काँच खान सक्छू र मलाई केहि नी हुन्‍न् । ' ],
            [ 'ᐊᓕᒍᖅ ᓂᕆᔭᕌᖓᒃᑯ ᓱᕋᙱᑦᑐᓐᓇᖅᑐᖓ' ]
        ];
    }

}